import { parseCustom } from './parse-custom'
import { isFirstPlayer, forceClickToPlay, isIE11 } from './functions'
import { adjustBehaviour } from './adjust-behaviour'

/* globals jwplayer */

/**
 * @function setup
 * initializes jwplayer
 * @param {Object} config - config with custom parameters for video.
 * @param {Object[]} promoted - videos that should be played in a playlist instead of requested video
 * @returns {*} - new jwplayer instance
 */
export function setup(config, promoted) {
	let playerInstance

	if (promoted && promoted.length) {
		playerInstance = setupPlaylist(config, promoted)
	} else {
		playerInstance = setupSingle(config)
	}

	adjustBehaviour(playerInstance)

	// for now disabled muted button
	// we will need it when we start selling
	// muted prerolls

	// setupVisuals(playerInstance)

	return playerInstance
}

function setupVisuals(player) {
	setupUnmuteIcon(player)
}

function setupUnmuteIcon(player) {
	const jwContainer = player.getContainer()
	const widgetContainer = jwContainer.closest('[data-videoid]')
	const unmuteBtn = widgetContainer.querySelector('.unmute-btn')

	player.on('mute', e => {
		unmuteBtn.style.display = e.mute ? 'flex' : 'none'
	})

	// checking if player is playing as mute initially
	// if so, triggering 'mute' event
	player.on('play', () => {
		if (player.getMute()) {
			player.trigger('mute', { mute: true })
		}
	})

	unmuteBtn.addEventListener('click', () => {
		player.setMute(false)
	})
}

export function setupPlaylist(config, videos) {
	const { custom, mediaid } = config
	const { repeat } = parseCustom(custom)

	return jwplayer(mediaid).setup({
		playlist: videos.map(v => getInitConfig(v)),
		repeat,
	})
}

export function setupSingle(config) {
	const { mediaid } = config

	const player = jwplayer(mediaid).setup(config)

	if (config.starttime) {
		setupStartTime(config, player)
	}

	return player
}

function setupStartTime(config, player) {
	if (config.starttime) {
		player.on('firstFrame', () => {
			player.seek(config.starttime)
		})
	}
}

/**
 * @function getInitConfig
 * @param custom {string}
 * @param id {string}
 * @param title {string}
 * @param autoplay {Number}
 * @param isEmbed {boolean}
 * @returns {*} config: argument for jwplayer.setup function
 */
export function getInitConfig({ id, title, autoplay, custom = '{}', isEmbed = false }) {
	const customArgs = parseCustom(custom)

	const defaultConfig = getDefaultConfig()
	const overridingConfig = {
		file: `https://content.jwplatform.com/manifests/${id}.m3u8`,
		playlist: `https://content.jwplatform.com/v2/media/${id}`,
		image: `https://content.jwplatform.com/thumbs/${id}.jpg`,
		mediaid: id,
		...customArgs,
	}

	// title is needed for 'upcoming videos' in playlist view
	if (title) {
		overridingConfig.title = title
	}

	// if autoplay prop is set, overriding autoplay
	if (autoplay !== undefined && autoplay !== null) {
		overridingConfig.autostart = !!autoplay
	}

	// Here we just need to override only 1 parameter in default config.
	if (customArgs.autoplaytimer) {
		overridingConfig.related = setCustomAutoplayTimer(defaultConfig, customArgs.autoplaytimer)
	}

	// do not force click-to-play for "viewable-autoplay" players
	if (isEmbed && !isFirstPlayer(id) && defaultConfig.autostart !== 'viewable') {
		forceClickToPlay(overridingConfig)
	}

	// because video is breaking all the ads
	// for ie11 was decided (TEMPORARY)
	// disable ads for video on ie11 until
	// JW fixes it
	if (isIE11() && isEmbed) {
		overridingConfig.advertising = null
	}

	const config = { ...defaultConfig, ...overridingConfig }
	return config
}

/**
 * @function
 * @returns Config specific for player
 * configured on JW admin
 */
function getDefaultConfig() {
	return { ...window.jwDefaults }
}

function setCustomAutoplayTimer(defaultConfig, autoplaytimer) {
	return { ...defaultConfig.related, autoplaytimer }
}
