import adBlockDetect from 'adblock-detect'
import createAdConfig from './create-ad-config'
import { isIE11 } from '../functions'

/**
 * @function - performing prebid logic
 * Docs: http://prebid.org/dev-docs/publisher-api-reference.html#module_pbjs.setConfig
 */
export default (mediaId, autoplay, withPrebid, isEmbedExternally) => {
	return new Promise((resolve, reject) => {
		if (!withPrebid || !playerHasAdsConfigured() || (isIE11() && isEmbedExternally)) resolve(null)

		try {
			adBlockDetect(adBlockFound => {
				if (adBlockFound) {
					reject(new Error('adblock found'))
				} else {
					prebid(resolve, reject, mediaId, autoplay)
				}
			})
		} catch (err) {
			reject(err)
		}
	}).catch(err => {
		// eslint-disable-next-line no-console
		console.error('Error occured. Skipping prebid', err)
		return new Promise(resolve => resolve())
	})
}

/**
 * @function
 * Performs logic regarding getting bids information,
 * forms url for getting ads
 * @param {*} resolve - callback to invoke if get any bids
 * @param {*} mediaId - id of requested video
 * @param {*} autoplay - if player is autoplayable
 */
function prebid(resolve, reject, mediaId, autoplay) {
	fallback()

	const { host } = window.location
	const { adUnit } = createAdConfig(host, mediaId, autoplay)

	window.pbjs.que.push(() => {
		window.pbjs.addAdUnits(adUnit)

		window.pbjs.setConfig(getConfig())

		window.pbjs.requestBids({
			adUnits: [adUnit],
			bidsBackHandler(bids) {
				try {
					if (!bids) {
						resolve(null)
					}

					const adUrl = window.pbjs.adServers.dfp.buildVideoUrl({
						adUnit,
						url: window.jwDefaults.advertising.schedule[0].tag,
					})

					resolve(adUrl)
				} catch (err) {
					reject(new Error(`error in bidsBackHandler ${err}`))
				}
			},
		})
	})
}

/**
 * @function
 * mocks prebid.js lib object if script is not loaded
 * yet
 */
function fallback() {
	window.pbjs = window.pbjs || {}
	window.pbjs.que = window.pbjs.que || []
}

/**
 * @function
 * @returns - config for Prebid.js library
 */
function getConfig() {
	const videoPriceGranularity = {
		buckets: [
			{
				min: 0,
				max: 1000,
				increment: 1,
			},
		],
	}

	return {
		// debug: true,
		cache: {
			url: 'https://prebid.adnxs.com/pbc/v1/cache',
		},
		bidderSequence: 'random',
		enableSendAllBids: false,
		currency: {
			adServerCurrency: 'NOK',
		},
		consentManagement: {
			cmpApi: 'iab',
			timeout: 100,
			allowAuctionWithoutConsent: true,
		},
		mediaTypePriceGranularity: {
			video: videoPriceGranularity,
		},
	}
}

/**
 * @returns {boolean}
 */
export function playerHasAdsConfigured() {
	if (!window.jwDefaults) {
		return false
	}
	const adConfig = window.jwDefaults.advertising
	return adConfig && adConfig.schedule && adConfig.schedule.length > 0
}
