/* eslint-disable react/state-in-constructor */
// Dependencies
import React from 'react'
import hoistStatics from 'hoist-non-react-statics'
import { createStore, applyMiddleware, compose } from 'redux'
import thunk from 'redux-thunk'
import { Provider } from 'react-redux'

import getDisplayName from '../utils/get-display-name'

/**
 * Higher order component factory
 * for adding redux
 * to your components.
 */
export default function getWithReduxDecorator({ reducer, initialState = {} }) {
	// Create a pass-through
	const logger = () => next => action => next(action)

	// Create developer tools middleware
	const devTools =
		typeof window === 'object' && window.__REDUX_DEVTOOLS_EXTENSION__ // eslint-disable-line no-underscore-dangle
			? window.__REDUX_DEVTOOLS_EXTENSION__() // eslint-disable-line no-underscore-dangle
			: f => f

	// Compose a store creator function with middleware
	const finalCreateStore = compose(
		applyMiddleware(thunk, logger),
		devTools,
	)(createStore)

	// Create a Redux store
	const makeStore = finalCreateStore.bind(null, reducer, initialState)

	// Return a function that extends a component with redux state handling
	return function withReduxDecorator(Component) {
		// Create a component that wraps our input component in a Redux <Provider>
		class withRedux extends React.Component {
			// Create a new Redux store for each component instance.
			// This avoids sharing state between instances
			state = {
				store: makeStore(),
			}

			// Render the component wrapped in a Redux <Provider>.
			// This exposes the store and gives children the ability to connect()
			render() {
				const { store } = this.state
				return (
					<Provider store={store}>
						<Component {...this.props} />
					</Provider>
				)
			}
		}
		// Add a specific display name
		withRedux.displayName = `${getDisplayName(Component)}WithRedux` // eslint-disable-line react/state-in-constructor

		// Return a decorated component with all the existing static methods hoisted
		return hoistStatics(withRedux, Component)
	}
}
