import React from 'react'
import PropTypes from 'prop-types'
import { format } from 'date-fns'
import { nb } from 'date-fns/locale'
import styled, { keyframes } from 'styled-components'

import { getColor } from '@aller/shiny/src/utils/get-color'
import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { LinkBarButton } from '@aller/shiny/src/atoms/LinkBarButton'
import { Kicker } from '@aller/shiny/src/atoms/Kicker'
import { Arrow } from '@aller/shiny/src/atoms/Arrow'
import { Image } from '@aller/shiny/src/atoms/Image'
import { VideoDescription, VideoPublished } from './styles'
import { slug } from '../../utils/slug'
import VideoWidget from './widgets'
import { VideoPlugHeading } from '../VideoHeading/VideoPlugHeading'

const horizontalHalf = props => `calc(${getVariable('horizontalBase')(props)} * 1 / 2)`

const SpoilerButton = styled(LinkBarButton)`
	&& {
		padding-left: ${horizontalHalf};
		padding-right: ${horizontalHalf};
		margin: 0;
		color: ${getColor('typeLight', 'light')};
		cursor: pointer;
		:focus {
			background-color: transparent;
		}
		span {
			font-size: ${getVariable('headingSmallSize')};
			@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.sm}em) {
				font-size: ${getVariable('headingRegularSize')};
			}
		}
	}
`

const fadeIn = keyframes`
	from {
		opacity: 0;
		transform: translateX(-3rem);
	}
	to {
		opacity: 1;
		transform: translateX(0);
	}
`

const StyledRootSection = styled.section`
	margin: 0;
`

const HidingBlock = styled.div`
	display: ${props => (props.isVisible ? 'block' : 'none')};
	padding-bottom: calc(${getVariable('verticalBase')} / 2);
	animation: ${fadeIn} 300ms cubic-bezier(0.165, 0.84, 0.44, 1);
	margin: 0 ${getVariable('horizontalBase')};
	@media (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		margin: 0;
	}
`

const ShowKicker = styled(Kicker)`
	text-align: left;
	padding-left: ${getVariable('horizontalBase')};
	font-size: ${getVariable('uiRegularSize')};
	font-style: italic;
	background-color: ${props => props.backgroundColor};
	color: ${props => props.fontColor};
`

const ShowImage = styled(Image)`
	width: calc(${getVariable('horizontalBase')} * 4);
	object-fit: cover;
`

const Meta = styled.div`
	display: flex;
	align-items: flex-start;
	padding: calc(${getVariable('verticalBase')} / 2) 0;
	& > :not(:last-child) {
		margin-right: calc(${getVariable('horizontalBase')});
		@media (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
			margin-right: calc(${getVariable('horizontalBase')} * 2);
		}
	}
	margin: 0 ${getVariable('horizontalBase')};
	@media (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		margin: 0;
	}
`

const VideoArrow = styled(Arrow)`
	vertical-align: ${p => (p.direction === 'up' ? 'sub' : 'text-top')};
	padding: 0.45rem;
	@media (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		vertical-align: ${p => (p.direction === 'up' ? 'sub' : 'super')};
		padding: 0.65rem;
	}
`

class VideoSection extends React.PureComponent {
	static formatPublished(publishedAt) {
		return format(new Date(publishedAt), 'd. MMMM yyyy', { locale: nb })
	}

	constructor(props) {
		super(props)
		const { title, description, publishedAt, show } = this.props
		this.state = {
			title,
			description,
			publishedAt,
			show,
			isVisible: false,
			arrowDirection: 'down',
		}
		this.toggleMeta = this.toggleMeta.bind(this)
	}

	componentDidMount() {
		if (!process.browser) return
		const that = this // this is for IE only

		const { id: mediaid } = this.props
		/* globals jwplayer */
		window.addEventListener(`playerLoaded-${mediaid}`, () => {
			jwplayer().on('playlistItem', playlistItem => {
				const { title, description, pubdate, mediaid: nextMediaId } = playlistItem.item
				const isMarketingPage = window.location.pathname.includes('/video/ad/')
				let slugPath = slug(title)
				if (isMarketingPage) {
					slugPath = 'ad'
				}
				// setting show to null because we can't receive show info from jw js api
				if (nextMediaId !== mediaid) {
					const stampDate = pubdate && new Date(pubdate * 1000)
					window.history.replaceState(playlistItem.item, null, `/video/${slugPath}/${nextMediaId}`)

					that.setState({
						title,
						description,
						publishedAt: stampDate,
						show: null,
					})
				}
			})
		})
	}

	toggleMeta() {
		const { isVisible } = this.state
		this.setState({
			isVisible: !isVisible,
			arrowDirection: isVisible ? 'down' : 'up',
		})
	}

	render() {
		const { title, description, publishedAt, isVisible, arrowDirection, show } = this.state
		const { showTitle } = this.props
		return (
			<StyledRootSection>
				{show && show.name && showTitle && (
					<ShowKicker backgroundColor={show.color} fontColor={show.textColor}>
						{show.name}
					</ShowKicker>
				)}
				<VideoWidget {...this.props} />
				<Meta>
					{show && show.icon && <ShowImage src={show.icon} alt={show.name} />}
					<VideoPlugHeading size="xlarge" marginTopFactor={0} marginBottomFactor={0}>
						{title}{' '}
						<SpoilerButton
							aria-label={`${isVisible ? 'Skjul' : 'Vis'} videobeskrivelse`}
							title={`${isVisible ? 'Skjul' : 'Vis'} videobeskrivelse`}
							useUnderline={false}
							activeBackground="transparent"
							onClick={this.toggleMeta}
						>
							<VideoArrow direction={arrowDirection} color="white" />
						</SpoilerButton>
					</VideoPlugHeading>
				</Meta>
				<HidingBlock isVisible={isVisible}>
					<VideoDescription>{description}</VideoDescription>
					{publishedAt && (
						<VideoPublished>{VideoSection.formatPublished(publishedAt)}</VideoPublished>
					)}
				</HidingBlock>
			</StyledRootSection>
		)
	}
}

VideoSection.propTypes = {
	id: PropTypes.string.isRequired,
	title: PropTypes.string.isRequired,
	description: PropTypes.string.isRequired,
	publishedAt: PropTypes.string.isRequired,
	custom: PropTypes.string,
	showTitle: PropTypes.bool,
	type: PropTypes.string,
	promoted: PropTypes.arrayOf(
		PropTypes.shape({
			id: PropTypes.string.isRequired,
			title: PropTypes.string.isRequired,
			custom: PropTypes.string,
		}),
	),
	show: PropTypes.shape({
		id: PropTypes.string.isRequired, // playlistId of the show
		icon: PropTypes.string,
		name: PropTypes.string,
		color: PropTypes.string, // validated on backend
		textColor: PropTypes.string, // validated on backend
	}),
}

VideoSection.defaultProps = {
	custom: null,
	promoted: [],
	show: null,
	showTitle: false,
	type: 'JW',
}

export default VideoSection
