/* eslint-disable import/prefer-default-export */
import React from 'react'
import propTypes from 'prop-types'
import styled from 'styled-components'

import { getColor } from '@aller/shiny/src/utils/get-color'
import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { BlockLink } from '@aller/shiny/src/atoms/BlockLink'
import { SmallRoundedButton, TinyRoundedButton } from '@aller/shiny/src/atoms/RoundedButtons'
import { FontIcon } from '@aller/shiny/src/atoms/FontIcon'

import mathRound from '../../../utils/math-round'

const StyledWeatherDropdown = styled.div`
	position: absolute;
	color: ${getColor('type')};
	background-color: ${getColor('white')};
	width: auto;
	z-index: 10;
	top: 5.5rem;
	left: 0;
	box-shadow: 0 0 4rem 0 rgba(0, 0, 0, 0.15);
	padding: calc(${getVariable('verticalBase')} / 2) calc(${getVariable('horizontalBase')} * 2)
		${getVariable('verticalBase')};
	@media screen and (min-width: ${getVariable('largeWidth')}) {
		width: 32rem;
	}
	::before {
		content: '';
		display: none;
		width: 1rem;
		height: 1rem;
		background-color: ${getColor('white')};
		transform: rotate(45deg);
		position: absolute;
		top: -5px;
		left: calc(${getVariable('verticalBase')});
		@media screen and (min-width: ${getVariable('largeWidth')}) {
			display: block;
		}
	}
	a {
		color: ${getColor('link')};
	}
	.weather-settings {
		line-height: 1;
		text-align: right;
		font-size: ${getVariable('uiSmallSize')};
	}
	.weather-label {
		font-size: ${getVariable('uiSmallSize')};
	}
	ul {
		padding: 0;
		margin: 0;
		:first-child {
			border: 0;
			margin-bottom: 0;
			padding-bottom: 0;
		}
		:last-child {
			border: 0;
		}
		li {
			display: flex;
			justify-content: space-between;
			align-items: center;
			margin-bottom: calc(${getVariable('verticalBase')} / 2);
			line-height: 2.4rem;
			border-bottom: 1px solid ${getColor('grayTint')};
			padding-bottom: calc(${getVariable('verticalBase')} / 2);
			.weather-location {
				flex: 1 0 40%;
			}
			.weather-icons {
				display: flex;
				align-items: center;
			}
			.weather-icons,
			.weather-below-zero,
			.weather-above-zero {
				flex: 1 0 20%;
			}
			.weather-location-btn {
				flex: 0 0 auto;
			}
			.weather-above-zero {
				color: ${getColor('primary')};
			}
			.weather-below-zero {
				color: ${getColor('secondaryDark')};
			}
		}
	}
`

const WeatherDropdown = ({ weather }) => {
	const weatherData = weather.map(data => {
		const { location: locationData, forecast, alerts } = data

		if (!locationData || !forecast) {
			return null
		}

		const url = `https://sol.no/vaer/${locationData.country}/${locationData.county}/${locationData.municipality}/${locationData.place}` // eslint-disable-line max-len
		const alertIcon = alerts ? (
			<img alt="OBS-varsel" src="https://sol.no/s/img/static/obs.png" />
		) : null
		const temperature = mathRound(forecast.tabular[0].temperature.value)
		const temperatureClass = temperature > 0 ? 'weather-above-zero' : 'weather-below-zero'

		return (
			<li key={`header-weather-${locationData.place}`}>
				<div className="weather-location">
					<span>{locationData.place}</span>
				</div>
				<div className="weather-icons">
					<img
						alt={forecast.tabular[0].symbol.name}
						src={
							'https://sol.no/s/img/h_30,w_30/static/weather_symbol/' +
							`${forecast.tabular[0].symbol.numberEx}.png`
						}
					/>
					{alertIcon}
				</div>
				<span className={temperatureClass}>
					{temperature}
					°C
				</span>
				<TinyRoundedButton className="weather-location-btn" href={url} suffixIconColor="link">
					Mer
					<FontIcon name="arrow-alt-right" size="0.8" />
				</TinyRoundedButton>
			</li>
		)
	})

	return (
		<StyledWeatherDropdown>
			<BlockLink className="weather-settings" href="https://sol.no/vaer/innstillinger">
				Endre sted
			</BlockLink>
			<ul>
				<li>
					<span className="weather-label">Valgt sted</span>
				</li>
				{weatherData}
			</ul>
			<SmallRoundedButton href="https://sol.no/vaer" className="weather-more-btn">
				Se været flere steder
			</SmallRoundedButton>
		</StyledWeatherDropdown>
	)
}

WeatherDropdown.propTypes = {
	weather: propTypes.any, // eslint-disable-line react/forbid-prop-types
}
WeatherDropdown.defaultProps = {
	weather: null,
}

export { WeatherDropdown }
