import React from 'react'
import PropTypes from 'prop-types'
import dynamic from 'next/dynamic'
import styled from 'styled-components'

import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { getColor } from '@aller/shiny/src/utils/get-color'

import { VerticalLinkBar } from '@aller/shiny/src/molecules/VerticalLinkBar'
import { LargeLinkBarDropdown } from '@aller/shiny/src/atoms/LinkBarDropdown'
import { LinkBarButton } from '@aller/shiny/src/atoms/LinkBarButton'
import { SvgIcon } from '@aller/shiny/src/atoms/SvgIcon'
import { HorizontalLinkBar } from '@aller/shiny/src/molecules/HorizontalLinkBar'
import { LinkBarLink, SmallLinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'

// This weird pattern is to make sure that:
// - The dynamic component is not included in the commons bundle
// - The component does not flash on client rerender
const ViLogo = dynamic(() =>
	import(/* webpackChunkName: "ViLogo" */ '@aller/shiny/src/atoms/ViLogo').then(c => c.ViLogo),
)

const VanillaDropdown = dynamic(() =>
	import(/* webpackChunkName: "VanillaDropdown" */ './Vanilla/VanillaDropdown'),
)

const desktopLinkProps = {
	textColor: 'secondary',
	useUnderline: false,
	ALLCAPS: true,
	activeBackgroundColor: 'secondary',
	activeTextColor: 'primary',
}

const Header = styled.header`
	padding-bottom: 0;
	@media (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		padding-bottom: 1rem;
	}
`

const mobileLinkProps = {
	textColor: 'primary',
	backgroundColor: 'secondary',
	activeBackgroundColor: 'primary',
	activeTextColor: 'secondary',
	useUnderline: false,
	ALLCAPS: true,
}

// positioned absolutely in favor of
// navlinks being aligned horizontally
const LogoLink = styled(LinkBarLink)`
	position: absolute;
	left: ${getVariable('horizontalBase')};
	top: calc(50% - 0.2rem); // aligning logo vertically
	transform: translate(0, -50%); // aligning logo vertically
	width: 11rem;
`

const StyledLink = styled(LinkBarLink)`
	& {
		margin-bottom: 0.5rem;
		&:hover {
			transition: all 0.1s ease-in-out;
		}

		@media (min-width: ${props => props.theme.flexboxgrid.breakpoints.lg}em) {
			padding-left: calc(${getVariable('horizontalBase')} * 2);
			padding-right: calc(${getVariable('horizontalBase')} * 2);
			margin-bottom: 0;
		}
	}
`

const links = [
	{ text: 'Historie', url: '//www.vi.no/emne/historie' },
	{ text: 'Teknologi', url: '//www.vi.no/teknologi' },
	{ text: 'Snekring', url: '//www.vi.no/emne/snekring' },
	{ text: 'Pensjon', url: '//www.vi.no/emne/pensjon' },
	{ text: 'Helse', url: '//www.vi.no/helse' },
	{ text: 'Arbeidsliv', url: '//www.vi.no/emne/arbeidsliv' },
	{ text: 'Hjernetrim', url: '//www.vi.no/emne/hjernetrim' },
]

const AlignedSvgIcon = styled(SvgIcon)`
	position: relative;
	top: 0.5rem;
`

const MenuDropdownTrigger = ({ hide, ...rest }) => (
	<LinkBarButton {...rest}>
		<SmallLinkBarLink textColor="secondary">meny</SmallLinkBarLink>
		<AlignedSvgIcon name="hamburger" size={2} opened={!hide} color="secondary" />
	</LinkBarButton>
)

const StyledMenuDropdown = styled(LargeLinkBarDropdown)`
	margin-left: auto;
	color: ${getColor('secondary')};
	text-transform: uppercase;
`

const HamburgerMenuDropdown = props => (
	<StyledMenuDropdown Trigger={MenuDropdownTrigger} {...props} />
)

const StyledVerticalLinkBar = styled(VerticalLinkBar)`
	& {
		padding: calc(${getVariable('verticalBase')} / 2) ${getVariable('horizontalBase')};
		width: 100%;
	}
`

const ViHeader = props => (
	<Header>
		<HorizontalLinkBar
			backgroundColor="primary"
			shouldFlexChildren
			justifyContent="space-between"
			isTopLevelComponent={false}
			overflow="visible"
		>
			<LogoLink
				marginRight={2}
				smSize={10}
				useUnderline={false}
				position="static"
				isBlockLink
				url="https://www.vi.no/"
				key={0}
			>
				<ViLogo />
			</LogoLink>
			<HorizontalLinkBar
				backgroundColor="transparent"
				shouldFlexChildren
				justifyContent="center"
				isTopLevelComponent={false}
				position="static"
				flex="1"
				md
				xs={false}
				key={1}
			>
				{links.map(link => {
					const { text, url } = link
					return <StyledLink size="large" {...desktopLinkProps} linkText={text} url={url} />
				})}
			</HorizontalLinkBar>
			<VanillaDropdown
				id="mobile-nav-dropdown"
				Dropdown={HamburgerMenuDropdown}
				linkText="Meny"
				position="static"
				md={false}
				key={2}
			>
				<StyledVerticalLinkBar backgroundColor="primary" hideArrow>
					{links.map(link => {
						const { text, url } = link
						return <StyledLink size="small" {...mobileLinkProps} linkText={text} url={url} />
					})}
				</StyledVerticalLinkBar>
			</VanillaDropdown>
		</HorizontalLinkBar>
	</Header>
)

MenuDropdownTrigger.propTypes = {
	hide: PropTypes.bool.isRequired,
}

export default ViHeader
