/* eslint-disable react/destructuring-assignment */

import React from 'react'
import propTypes from 'prop-types'
import styled from 'styled-components'

import { getColor } from '@aller/shiny/src/utils/get-color'
import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { FontIcon } from '@aller/shiny/src/atoms/FontIcon'

const KvasirSearch = styled.div`
	display: flex;
`

const SearchForm = styled.form`
	display: none;
	position: absolute;
	flex: 0 0 8rem;
	flex-direction: row;
	justify-content: space-between;
	top: 0.3rem;
	right: 3.2rem;
	opacity: 1;
	margin: 0 ${getVariable('horizontalBase')} 0 calc(${getVariable('horizontalBase')} / 2);
	padding: 0;
	height: 3.5rem;
	border-radius: 2rem;
	background: ${getColor('secondary', 'dark')};
	transition: all 0.2s ease-in-out;

	@media (min-width: ${getVariable('largeWidth')}) {
		display: flex;
		position: static;
		flex: 0 0 16.4rem;
		background: ${getColor('white')};
	}
`

const SearchInput = styled.input`
	display: flex;
	margin: 0 0 0 2rem;
	padding: 0;
	height: 3.4rem;
	border: 0;
	border-radius: 0;
	font-size: ${getVariable('uiRegularSize')};

	width: ${props => props.defaultWidth};
	background: ${getColor('secondary', 'dark')};
	color: ${getColor('white')};

	vertical-align: bottom;
	transition: width 0.2s ease-in-out;

	&::placeholder {
		color: ${getColor('grayTintLight')};
	}

	&:focus {
		border: 0;
		outline: none;
	}

	@media (min-width: ${getVariable('largeWidth')}) {
		width: 14.8rem;
		background: ${getColor('white')};
		color: ${getColor('type')};

		&::placeholder {
			color: initial;
		}
	}
`

const Submit = styled.button`
	display: flex;
	justify-content: center;
	width: 2.9rem;
	height: 2.9rem;
	padding: 0;
	border: 0;
	border-radius: 1.45rem;
	background-color: ${getColor('secondary', 'dark')};
	color: ${getColor('white')};
	font-size: 2.5rem;
	vertical-align: bottom;
	margin: 0.3rem 0.4rem;
	transition: opacity 0.2s ease-in-out;

	@media (min-width: ${getVariable('largeWidth')}) {
		background-color: ${getColor('primary')};

		&:hover {
			cursor: pointer;
			opacity: 0.8;
		}
	}
`

const RevealSearch = styled(Submit)`
	width: 3.6rem;
	height: 3.5rem;
	border-radius: 2rem;

	@media (min-width: ${getVariable('largeWidth')}) {
		display: none;
	}
`

class KvasirSearchField extends React.Component {
	constructor(props) {
		super(props)
		this.state = { displaySearch: false }
		this.lpPrefix = props.lpPrefix
		this.searchForm = React.createRef()
		this.searchInput = React.createRef()
		this.toggleSearch = this.toggleSearch.bind(this)
	}

	toggleSearch = () => {
		let { displaySearch } = this.state
		displaySearch = !displaySearch

		this.setState({ displaySearch }, () => {
			const searchForm = this.searchForm.current
			const searchInput = this.searchInput.current
			if (displaySearch) {
				searchForm.style.display = 'flex'
				searchInput.focus()
			} else {
				searchForm.style.display = 'none'
				searchInput.blur()
			}
		})
	}

	render() {
		const { displaySearch } = this.state
		const { defaultWidth } = this.props

		return (
			<KvasirSearch>
				<SearchForm ref={this.searchForm} action={`${this.lpPrefix}https://www.kvasir.no/alle`}>
					<input type="hidden" name="partnerid" value="sol" />
					<input type="hidden" name="source" value="search" />
					<SearchInput
						ref={this.searchInput}
						type="text"
						name="q"
						aria-label="Søk i Kvasir"
						placeholder="Søk i Kvasir"
						defaultWidth={defaultWidth}
					/>
					<Submit aria-label="Søk" type="submit">
						<FontIcon name="search" size="1.8" />
					</Submit>
				</SearchForm>

				<RevealSearch aria-label="Vis søkefelt" type="button" onClick={this.toggleSearch}>
					{!displaySearch && <FontIcon name="search" size="1.8" />}
					{displaySearch && <FontIcon name="close" size="1.8" />}
				</RevealSearch>
			</KvasirSearch>
		)
	}
}

KvasirSearchField.propTypes = {
	lpPrefix: propTypes.string,
	defaultWidth: propTypes.string,
}

KvasirSearchField.defaultProps = {
	lpPrefix: '',
	defaultWidth: '18rem',
}

export default KvasirSearchField
