import React, { memo } from 'react'
import PropTypes from 'prop-types'
import styled, { css } from 'styled-components'
import { useInView } from 'react-intersection-observer'

import { KKLogo } from '@aller/shiny/src/atoms/KKLogo'
import { FullWidthLinkBar } from '@aller/shiny/src/molecules/FullWidthLinkBar'
import { HorizontalLinkBar } from '@aller/shiny/src/molecules/HorizontalLinkBar'
import { VerticalLinkBar } from '@aller/shiny/src/molecules/VerticalLinkBar'
import { LinkBarItem } from '@aller/shiny/src/atoms/LinkBarItem'
import { LinkBarHeading } from '@aller/shiny/src/atoms/LinkBarHeading'
import { LinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'
import { LinkBarSearchField } from '@aller/shiny/src/atoms/LinkBarSearchField'
import { LinkBarDropdown } from '@aller/shiny/src/atoms/LinkBarDropdown'
import { getVariable } from '@aller/shiny/src/utils/get-variable'

import UserDropDown from '../UserDropDown'

import Trigger from './Trigger'

const Banner = styled.div`
	background-image: linear-gradient(-90deg, rgba(223, 205, 192, 0.9), rgba(243, 227, 205, 0.9));
	height: 13rem;
	background-color: #ebb2b2;
	position: relative;
	display: flex;
	justify-content: center;
	align-items: center;
`

const BannerLink = styled.a`
	background-color: #222;
	border-radius: 50%;
	height: 10rem;
	width: 10rem;
	display: flex;
	justify-content: center;
	align-items: center;
`

const BannerLogo = styled.svg`
	height: 6rem;
	width: 6rem;
`

const Header = styled.header`
	position: sticky;
	top: 0;
	margin-bottom: 1rem;
	z-index: 2;
	border-bottom: 0.1rem solid #eee;
	height: 6.1rem;
`

const RightListItem = styled.li`
	position: absolute;
	top: 0;
	right: 0;
	width: 100%;
`

const Link = styled(LinkBarLink)`
	font-size: ${getVariable('uiSmallSize')};
	line-height: 3.6rem;
	font-weight: 700;
`

const DDLink = styled(LinkBarLink)`
	font-size: 1.4rem;
	font-weight: 300;
	line-height: 1.5rem;
`

// We need the custom Trigger to recreate the non-standard styles in the other
// LinkBar* elements in the menu
const Dropdown = props => <LinkBarDropdown Trigger={Trigger} {...props} />

const Search = styled(LinkBarSearchField)`
	&& {
		max-width: 7rem;

		&:focus {
			max-width: 30rem;
		}

		&[value!=''] {
			max-width: 30rem;
		}
	}
`

const CustomItem = styled(LinkBarItem)`
	flex: 1 1 50%;
	padding-left: calc(1 / 2 * ${getVariable('horizontalBase')});
	padding-right: calc(1 / 2 * ${getVariable('horizontalBase')});
`
const CustomLink = styled(LinkBarLink)`
	font-weight: ${getVariable('uiWeight')};
	display: block;
	width: 100%;
	text-align: center;
`
const KKLink = props => (
	<CustomItem>
		<CustomLink {...props} />
	</CustomItem>
)

const CustomHeading = styled(LinkBarHeading)`
	font-weight: ${getVariable('uiWeightBold')};
	padding-left: ${getVariable('horizontalBase')};
	padding-right: ${getVariable('horizontalBase')};
`

const mobileLinkProps = {
	isListItem: true,
	useUnderline: false,
	size: 'large',
	inset: true,
	backgroundColor: 'grayTintLight',
	activeBackgroundColor: 'grayTintLight',
}

const linkProps = {
	activeBackgroundColor: 'grayTintLight',
	useUnderline: false,

	ALLCAPS: true,
}

const userDropDownProps = {
	useUnderline: false,
}

const ddLinkProps = {
	useUnderline: false,
	activeBackgroundColor: 'grayTintLight',
}

const CircleA = styled.a`
	display: block;
	position: absolute;
	width: 6rem;
	height: 6rem;
	left: calc(1 / 2 * ${getVariable('horizontalBase')});
	z-index: 4;

	${p =>
		p.hide &&
		css`
			display: none;
		`}
`

const Fullbar = memo(({ displayDropdownsInitially }) => (
	<FullWidthLinkBar
		background="linear-gradient(0deg, #fff, #fff .1rem)"
		overflow="visible"
		justifyContent="center"
		isTopLevelComponent={false}
		shouldFlexChildren
	>
		<Dropdown
			{...linkProps}
			linkText="Innhold"
			xs={false}
			sm
			lg={false}
			displayInitially={displayDropdownsInitially}
		>
			<VerticalLinkBar background="white">
				<DDLink {...ddLinkProps} url="/livet/" linkText="Livet mitt" />
				<DDLink {...ddLinkProps} url="/mote/" linkText="Mote og skjønnhet" />
				<DDLink {...ddLinkProps} url="/helse/" linkText="Helse" />
				<DDLink {...ddLinkProps} url="/underholdning/" linkText="Underholdning" />
				<DDLink {...ddLinkProps} url="/mamma/" linkText="Mamma" />
				<DDLink {...ddLinkProps} url="/horoskop/" linkText="Horoskop" />
				<DDLink {...ddLinkProps} url="/meninger/" linkText="Meninger" />
				<DDLink {...ddLinkProps} url="/bolig/" linkText="Interiør" />
				<DDLink {...ddLinkProps} url="/reise/" linkText="Reise" />
				<DDLink {...ddLinkProps} url="/mat/" linkText="Mat" />
				<DDLink {...ddLinkProps} url="/tips/" linkText="Praktiske tips" />
				<DDLink {...ddLinkProps} url="/video/" linkText="Video" />
				<CustomHeading size="small" ALLCAPS>
					Tjenester
				</CustomHeading>
				<DDLink {...ddLinkProps} url="https://allerlive.no/kk-mila/" linkText="KK-mila" />
				<DDLink {...ddLinkProps} url="/bli-ny/" linkText="Bli ny" />
				<DDLink {...ddLinkProps} url="/rabattkoder/" linkText="Rabattkoder" />
				<DDLink
					{...ddLinkProps}
					url="https://www.allershopping.no/abokampanje-kkno/"
					linkText="Abonner på KK"
				/>
			</VerticalLinkBar>
		</Dropdown>
		<Link {...linkProps} xs={false} lg url="/livet/" linkText="Livet mitt" />
		<Link {...linkProps} xs={false} lg url="/mote/" linkText="Mote og skjønnhet" />
		<Link {...linkProps} xs={false} lg url="/helse/" linkText="Helse" />
		<Link {...linkProps} xs={false} lg url="/underholdning/" linkText="Underholdning" />
		<Link {...linkProps} xs={false} lg url="/mamma/" linkText="Mamma" />
		<Dropdown
			{...linkProps}
			xs={false}
			lg
			linkText="Mer"
			displayInitially={displayDropdownsInitially}
		>
			<VerticalLinkBar background="white">
				<DDLink {...ddLinkProps} url="/horoskop/" linkText="Horoskop" />
				<DDLink {...ddLinkProps} url="/meninger/" linkText="Meninger" />
				<DDLink {...ddLinkProps} url="/bolig/" linkText="Interiør" />
				<DDLink {...ddLinkProps} url="/reise/" linkText="Reise" />
				<DDLink {...ddLinkProps} url="/mat/" linkText="Mat" />
				<DDLink {...ddLinkProps} url="/tips/" linkText="Praktiske tips" />
				<DDLink {...ddLinkProps} url="/video/" linkText="Video" />
				<CustomHeading size="small" ALLCAPS>
					Tjenester
				</CustomHeading>
				<DDLink {...ddLinkProps} url="https://allerlive.no/kk-mila/" linkText="KK-mila" />
				<DDLink {...ddLinkProps} url="/bli-ny/" linkText="Bli ny" />
				<DDLink {...ddLinkProps} url="/rabattkoder/" linkText="Rabattkoder" />
				<DDLink
					{...ddLinkProps}
					url="https://www.allershopping.no/abokampanje-kkno/"
					linkText="Abonner på KK"
				/>
			</VerticalLinkBar>
		</Dropdown>
		<RightListItem isListItem>
			<HorizontalLinkBar
				backgroundColor="white"
				shouldFlexChildren
				overflow="visible"
				isTopLevelComponent={false}
				justifyContent="flex-end"
			>
				<UserDropDown linkProps={userDropDownProps} />
				<Search
					size="large"
					backgroundColor="white"
					fullWidth
					placeholder="Søk"
					action="https://www.kk.no/tekstarkiv/"
					xs={false}
					sm
				/>
				<Dropdown
					{...linkProps}
					position="static"
					linkText="Meny"
					sm={false}
					displayInitially={displayDropdownsInitially}
				>
					<VerticalLinkBar backgroundColor="white" align="right">
						<LinkBarSearchField
							fullWidth
							size="large"
							backgroundColor="primary"
							textColor="white"
							icon="search"
							iconColor="white"
							placeholder="Søk"
							action="https://www.kk.no/tekstarkiv/"
						/>
						<CustomHeading size="small" ALLCAPS>
							Redaksjonelt innhold
						</CustomHeading>
						<HorizontalLinkBar
							shouldWrap
							shouldFlexChildren
							backgroundColor="transparent"
							isTopLevelComponent={false}
						>
							<KKLink {...mobileLinkProps} linkText="Helse" url="/helse/" />
							<KKLink {...mobileLinkProps} linkText="Underholdning" url="/underholdning/" />
							<KKLink {...mobileLinkProps} linkText="Livet mitt" url="/livet/" />
							<KKLink {...mobileLinkProps} linkText="Mote og skjønnhet" url="/mote/" />
							<KKLink {...mobileLinkProps} linkText="Mamma" url="/mamma/" />
							<KKLink {...mobileLinkProps} linkText="Horoskop" url="/horoskop/" />
							<KKLink {...mobileLinkProps} linkText="Meninger" url="/meninger/" />
							<KKLink {...mobileLinkProps} linkText="Interiør" url="/bolig/" />
							<KKLink {...mobileLinkProps} linkText="Reise" url="/reise/" />
							<KKLink {...mobileLinkProps} linkText="Mat" url="/mat/" />
							<KKLink {...mobileLinkProps} linkText="Praktiske tips" url="/tips/" />
							<KKLink {...mobileLinkProps} linkText="Video" url="/video/" />
						</HorizontalLinkBar>
						<CustomHeading size="small" ALLCAPS>
							Tjenester
						</CustomHeading>
						<HorizontalLinkBar shouldWrap shouldFlexChildren backgroundColor="transparent">
							<KKLink {...mobileLinkProps} linkText="KK-mila" url="/kk-mila/" />
							<KKLink {...mobileLinkProps} linkText="Bli ny" url="/bli-ny/" />
							<KKLink {...mobileLinkProps} linkText="Shopping" url="/shopping/" />
							<KKLink {...mobileLinkProps} linkText="Rabattkoder" url="/rabattkoder/" />
							<KKLink
								{...mobileLinkProps}
								linkText="Abonner på KK"
								url="//www.allershopping.no/abokampanje-kkno/"
							/>
						</HorizontalLinkBar>
					</VerticalLinkBar>
				</Dropdown>
			</HorizontalLinkBar>
		</RightListItem>
	</FullWidthLinkBar>
))
Fullbar.propTypes = {
	displayDropdownsInitially: PropTypes.bool,
}
Fullbar.defaultProps = {
	displayDropdownsInitially: false,
}

const KKHeader = props => {
	const { displayDropdownsInitially } = props
	const [ref, inView] = useInView()

	let server = true
	if (process.browser) {
		server = false
	}

	/* eslint-disable max-len */
	return (
		<>
			<Banner ref={ref}>
				<BannerLink href="//www.kk.no" title="Forsiden">
					<BannerLogo width="28" height="22" viewBox="-1 0 28 22">
						<g fill="#FFF" fillRule="evenodd">
							<path d="M3.825.768c.148 0 .268.12.268.267V21.55c0 .147-.12.267-.268.267H.59a.269.269 0 0 1-.268-.267V1.035C.322.888.443.768.59.768h3.235M4.38 14.405a.373.373 0 0 1 .026-.425l4.161-4.804c.258-.293.571-.46.77-.54.185-.075.506-.186.91-.186h3.194c.148 0 .189.09.091.201l-4.897 5.545 5.033 7.406c.088.119.04.215-.108.215h-2.635c-.612 0-1.02-.11-1.364-.271-.33-.155-.672-.424-.999-.885L4.38 14.405M17.46.768c.148 0 .268.12.268.267V21.55c0 .147-.12.267-.268.267h-3.234a.269.269 0 0 1-.269-.267V1.035c0-.147.121-.267.269-.267h3.234M18.015 14.405a.373.373 0 0 1 .026-.425l4.162-4.804a2.08 2.08 0 0 1 .77-.54c.185-.075.505-.186.91-.186h3.194c.147 0 .188.09.09.201l-4.896 5.545 5.032 7.406c.088.119.04.215-.108.215H24.56c-.612 0-1.02-.11-1.364-.271-.33-.155-.672-.424-.998-.885l-4.183-6.256" />
						</g>
					</BannerLogo>
				</BannerLink>
			</Banner>
			<Header>
				<Fullbar displayDropdownsInitially={displayDropdownsInitially} />
				<h1>
					<CircleA href="//www.kk.no" title="Forsiden" hide={inView || server}>
						<KKLogo hasCircle />
					</CircleA>
				</h1>
			</Header>
		</>
	)
}

KKHeader.whyDidYouRender = true
KKHeader.propTypes = {
	displayDropdownsInitially: PropTypes.bool,
}
KKHeader.defaultProps = {
	displayDropdownsInitially: false,
}

export default KKHeader
