import React from 'react'
import dynamic from 'next/dynamic'
import styled from 'styled-components'

import { LinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'
import { BlockLink } from '@aller/shiny/src/atoms/BlockLink'
import { Paragraph } from '@aller/shiny/src/atoms/Paragraph'
import { HorizontalLinkBar } from '@aller/shiny/src/molecules/HorizontalLinkBar'
import { getColor } from '@aller/shiny/src/utils/get-color'
import { getVariable } from '@aller/shiny/src/utils/get-variable'

const Elbil24Logo = dynamic(() =>
	import(/* webpackChunkName: "Elbil24Logo" */ '@aller/shiny/src/atoms/Elbil24Logo').then(
		c => c.Elbil24Logo,
	),
)

const TopNavigation = styled.div`
	background: ${getColor('primary')};
	display: flex;
	flex-direction: column;
	align-items: center;
	font-family: ${getVariable('mainFont')};
`

const LogoContainer = styled.div`
	display: flex;
	flex-direction: column;
	align-items: center;
	margin-top: 4.4rem;
	svg {
		width: 19rem;
		@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
			width: 23rem;
		}
	}
`

const Slogan = styled(Paragraph)`
	color: ${getColor('white')};
	font-weight: ${getVariable('uiWeightSemiBold')};
	padding-top: 1rem;
`

const Branding = styled.div`
	display: flex;
`

const MenuScroll = styled.div`
	position: relative;
	display: flex;
	flex-direction: row;
	justify-content: flex-start;
	align-items: center;
	background: ${getColor('secondary')};
	scroll-behavior: smooth;
	overflow-y: hidden;
	overflow-x: auto;
	-webkit-overflow-scrolling: touch;
	-ms-overflow-style: -ms-autohiding-scrollbar;
	height: 5rem;
	z-index: 9999;

	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		justify-content: center;
	}
`

const Menu = styled(HorizontalLinkBar)`
	display: flex;
	flex-direction: row;
	list-style: none;
	justify-content: space-around;
	min-width: 80rem;
	max-width: 128rem;

	> li {
		padding: 0 2rem;
	}
`

const linkProps = {
	useUnderline: false,
	activeBackgroundColor: 'transparent',
	textColor: 'white',
	ALLCAPS: true,
}

const MenuLink = styled(LinkBarLink)`
	& {
		font-family: ${getVariable('mainFont')};
		font-weight: ${getVariable('uiWeightBold')};
		font-style: italic;
		font-size: ${getVariable('headingMediumSize')};

		transition: all 0.2s ease-out;
		border-bottom: 0.2rem solid transparent;
		padding: 0;

		:hover {
			color: ${getColor('grayTint', 'light')};
			transition: all 0.2s ease-out;
			border-bottom-color: ${getColor('white')};
		}
	}
`

const Elbil24Header = () => (
	<header>
		<TopNavigation>
			<Branding>
				<BlockLink href="https://www.elbil24.no">
					<LogoContainer>
						<Elbil24Logo pro />
						<Slogan>Norges ledende nettavis om elbiler</Slogan>
					</LogoContainer>
				</BlockLink>
			</Branding>
		</TopNavigation>
		<MenuScroll>
			<Menu
				isTopLevelComponent // Use the whole width
				backgroundColor="secondary"
				overflow="visible"
				shouldFlexChildren
			>
				<MenuLink {...linkProps} linkText="Pro" url="https://www.elbil24.no/pro" />
				<MenuLink {...linkProps} linkText="Tester" url="https://www.elbil24.no/tester" />
				<MenuLink {...linkProps} linkText="Video" url="https://www.elbil24.no/video" />
				<MenuLink {...linkProps} linkText="Nyttig" url="https://www.elbil24.no/nyttig" />
				<MenuLink {...linkProps} linkText="Lading" url="https://www.elbil24.no/lading" />
				<MenuLink {...linkProps} linkText="Nyhetsbrev" url="https://www.elbil24.no/nyhetsbrev" />
				<MenuLink
					{...linkProps}
					linkText="Medlemsfordeler"
					url="https://www.elbil24.no/medlemsfordeler"
				/>
			</Menu>
		</MenuScroll>
	</header>
)

export default Elbil24Header
