import React from 'react'
import PropTypes from 'prop-types'
import styled from 'styled-components'
import dynamic from 'next/dynamic'
import { HorizontalLinkBar } from '@aller/shiny/src/molecules/HorizontalLinkBar'
import { Header } from '@aller/shiny/src/atoms/Header'
import { getVariable, getColor } from '@aller/shiny/src/utils'
import { LargeLinkBarDropdown } from '@aller/shiny/src/atoms/LinkBarDropdown'
import { LinkBarItem } from '@aller/shiny/src/atoms/LinkBarItem'
import { LinkBarLink as DefLinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'
import { LargeLinkBarSearchField } from '@aller/shiny/src/atoms/LinkBarSearchField'
import { VerticalLinkBar } from '@aller/shiny/src/molecules/VerticalLinkBar'
import { Dot as DefaultDot } from '@aller/shiny/src/atoms/Dot'
import { Trigger as DesktopTrigger, MobileMenuTrigger } from './Trigger'
import DesktopMenu from './DesktopMenu'
import links from './links'
import UserDropDown from './UserDropDown'
import MobileMenu from './MobileMenu'
import MobileMenuLink from './MobileMenuLink'
import withBorder from './withBorder'

const linkProps = {
	useUnderline: false,
	activeBackgroundColor: 'primaryDark',
	textColor: 'white',
	activeTextColor: 'white',
	ALLCAPS: true,
}

const linkPropsDark = {
	useUnderline: false,
	ALLCAPS: false,
	textColor: 'darkness',
	activeTextColor: 'darkness',
	size: 'large',
}
const LinkBarLink = styled(DefLinkBarLink)`
	transition: 0.2s;
	font-size: ${getVariable('uiSmallSize')};
	&:hover {
		transition: 0.2s;
	}
`
const boxShadow = '0 3.5rem 5rem 0 rgba(0, 0, 0, 0.4)'
const Dropdown = styled(LargeLinkBarDropdown)`
	&& {
		max-height: 4.8rem;
	}
`
const Dot = styled(DefaultDot)`
	&& {
		margin-right: ${getVariable('horizontalBase')};
		width: 1.1rem;
		height: 1.1rem;
	}
`
const DagbladetLogo = dynamic(() =>
	import(/* webpackChunkName: "DagbladetLogo" */ '@aller/shiny/src/atoms/DagbladetLogo').then(
		c => c.DagbladetLogo,
	),
)
const LogoLink = styled(LinkBarLink)`
	&& {
		max-width: 15.2rem;
		height: 4.8rem;
		margin-left: -0.2rem;
		padding-bottom: 0.2rem;
		margin-top: 0;
		@media screen and (max-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
			max-width: 11rem;
			margin-left: 0.2rem;
			height: calc(${getVariable('verticalBase')} * 2);
		}
		& > svg {
			transition: 0.2s;
			padding-left: 0.5rem;
			:hover {
				background-color: ${getColor('primaryDark')};
				transition: 0.2s;
			}
		}
	}
`
const VanillaDropdown = dynamic(() =>
	import(/* webpackChunkName: "VanillaDropdown" */ '../Vanilla/VanillaDropdown'),
)
const StyledHeader = styled(Header)`
	&& {
		position: fixed;
		width: 100%;
	}
`

const LinkBarLinkBold = styled(LinkBarLink)`
	&& {
		font-weight: ${getVariable('uiWeightBold')};
	}
`

const StyledLinkBarSearchField = styled(LargeLinkBarSearchField)`
	height: 3.8rem;
	background-color: #fff;
`
const SearchFieldDynamicWidth = styled(LargeLinkBarSearchField)`
	&& {
		height: 3rem;
		width: 8rem;
		background-color: #fff;
		:focus {
			position: absolute;
			right: 0;
			max-width: 45rem;
			width: 100%;
			z-index: 200;
		}
		@media screen and (max-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
			display: none;
		}
	}
`
const StyledVerticalLinkBar = styled(VerticalLinkBar)`
	&& {
		width: 100vw !important;
		ul {
			position: initial;
			box-shadow: none;
			&:before {
				display: none;
			}
		}
	}
`

const StyledHorizontalLinkBarWithPadding = styled(HorizontalLinkBar)`
	padding: 0;
	padding-bottom: calc(${getVariable('verticalBase')} / 2);
`
const LinkBarLinkDynamicBold = styled(LinkBarLink)`
	padding: calc(1 * 1 / 2 * (4 / 2 * 2.4rem - 2.4rem)) 1.4rem; // get From LinkBarElementBase
	&& {
		@media screen and (min-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
			font-weight: ${getVariable('uiWeightBold')};
		}
		@media screen and (max-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
			padding-right: 0.7rem;
		}
	}
`

const StyledLinkBarLinkBold = styled(LinkBarLinkBold)`
	width: 100%;
	text-align: center;
	padding: calc(${getVariable('verticalBase')} / 2) 0;
`
const StyledLinkBarLinkBoldwithBorder = withBorder(['right'])(StyledLinkBarLinkBold)

const GrowingListItem = styled(LinkBarItem)`
	flex: 1 0 auto;
`
const TopAlignBar = styled(HorizontalLinkBar)`
	flex: 1 0 auto;
	align-items: flex-start;
	padding: 0;
`
const MobileVerticalLinkBar = styled(VerticalLinkBar)`
	&& {
		height: 80vh;
		overflow-y: auto;
		flex-wrap: nowrap;
	}
`
const DagbladetHeader = ({ displayDropdownsInitially, user }) => (
	<StyledHeader>
		<TopAlignBar
			backgroundColor="primary"
			isTopLevelComponent={false} // Use the full width
			shouldFlexChildren
			justifyContent="space-between"
			overflow="visible"
			zIndex={200}
		>
			<GrowingListItem isListItem>
				<TopAlignBar
					backgroundColor="transparent"
					isTopLevelComponent={false}
					overflow="visible"
					shouldFlexChildren
				>
					<LogoLink {...linkProps} useUnderline={false} isBlockLink url="/" title="Dagbladet">
						<DagbladetLogo />
					</LogoLink>
					<LinkBarLink {...linkProps} xs={false} sm linkText="Nyheter" url="/nyheter" />
					<LinkBarLink {...linkProps} xs={false} md linkText="Sport" url="/sport" />
					<LinkBarLink {...linkProps} xs={false} lg linkText="Kjendis" url="/kjendis" />
					<LinkBarLink {...linkProps} xs={false} lg linkText="TV" url="/video" />
					<LinkBarLink {...linkProps} xs={false} lg linkText="Ski-VM" url="/ski-vm-2019" />

					<VanillaDropdown
						{...linkProps}
						Dropdown={Dropdown}
						displayInitially={displayDropdownsInitially}
						id="commercial-dropdown"
						xs={false}
						sm
						linkText="Innhold"
						position="static"
						activeBackgroundColor="primary"
						Trigger={DesktopTrigger}
					>
						<StyledVerticalLinkBar
							shouldFadeOut
							hideArrow
							boxShadow={boxShadow}
							backgroundColor="white"
						>
							{[<DesktopMenu key={0} data={links} />]}
						</StyledVerticalLinkBar>
					</VanillaDropdown>
				</TopAlignBar>
			</GrowingListItem>
			<LinkBarLinkBold
				{...linkProps}
				xs={false}
				lg
				ALLCAPS
				linkText="Tips Oss"
				url="https://www.dagbladet.no/info"
			/>
			<LinkBarLinkBold
				{...linkProps}
				xs={false}
				md
				linkText="Kjøp Pluss!"
				url="https://www.dagbladet.no/pluss-om"
			/>
			<LinkBarLinkDynamicBold
				id="dagbladet-login-button"
				{...linkProps}
				linkText="Logg Inn"
				url="//www.dagbladet.no/app/minside"
			/>
			<UserDropDown selector="dagbladet" />
			<VanillaDropdown
				Dropdown={Dropdown}
				id="xs-menu"
				displayInitially={displayDropdownsInitially}
				{...linkProps}
				xs
				sm={false}
				position="static"
				linkText="Meny"
				textColor="white"
				Trigger={MobileMenuTrigger}
				zIndex={7}
				activeBackgroundColor="transparent"
			>
				<MobileVerticalLinkBar
					align="right"
					hideArrow
					boxShadow={boxShadow}
					backgroundColor="white"
				>
					<StyledHorizontalLinkBarWithPadding
						backgroundColor="primary"
						shouldFlexChildren
						isTopLevelComponent={false}
					>
						<StyledLinkBarSearchField
							backgroundColor="white"
							isListItem
							fullWidth
							placeholder="Søk"
							action="https://www.dagbladet.no/tekstarkiv/"
						/>
					</StyledHorizontalLinkBarWithPadding>
					<HorizontalLinkBar
						backgroundColor="grayTintLighter"
						shouldFlexChildren
						isTopLevelComponent={false}
					>
						<StyledLinkBarLinkBoldwithBorder
							isListItem
							xs={6}
							{...linkPropsDark}
							url="//www.dagbladet.no/info"
							linkText="Tips oss"
						/>
						<StyledLinkBarLinkBold
							isListItem
							xs={6}
							{...linkPropsDark}
							url="//www.dagbladet.no/pluss-om"
							linkText="Kjøp pluss!"
						/>
					</HorizontalLinkBar>
					<MobileMenuLink {...linkPropsDark} url="https://www.dagbladet.no/nyheter">
						<Dot size="small" />
						Nyheter
					</MobileMenuLink>
					<MobileMenuLink {...linkPropsDark} url="//www.dagbladet.no/sport">
						<Dot color="blue" size="small" />
						Sport
					</MobileMenuLink>
					<MobileMenuLink {...linkPropsDark} linkText="" url="//www.dagbladet.no/kjendis">
						<Dot color="green" size="small" />
						Kjendis
					</MobileMenuLink>
					<MobileMenuLink {...linkPropsDark} url="//www.dagbladet.no/video">
						<Dot color="navy" size="small" />
						TV
					</MobileMenuLink>
					<MobileMenuLink {...linkPropsDark} url="//www.dagbladet.no/pluss">
						<Dot color="navy" size="small" />
						Pluss
					</MobileMenuLink>
					<MobileMenu data={links} />
				</MobileVerticalLinkBar>
			</VanillaDropdown>
			<SearchFieldDynamicWidth
				isListItem
				xs={false}
				md
				action="https://www.dagbladet.no/tekstarkiv/"
				placeholder="Søk"
				backgroundColor="grayTintLighter"
				iconColor="typeLight"
			/>
		</TopAlignBar>
	</StyledHeader>
)
DagbladetHeader.propTypes = {
	displayDropdownsInitially: PropTypes.bool,
	preventWeatherUpdate: PropTypes.bool,
	user: PropTypes.shape({
		user_id: PropTypes.oneOfType([PropTypes.string, PropTypes.number]).isRequired,
		name: PropTypes.string.isRequired,
	}),
}
DagbladetHeader.defaultProps = {
	displayDropdownsInitially: false,
	preventWeatherUpdate: false,
	user: undefined,
}

export default DagbladetHeader
