import React from 'react'
import PropTypes from 'prop-types'
import styled from 'styled-components'
import { SmallLinkBarHeading } from '@aller/shiny/src/atoms/LinkBarHeading'
import { Grid } from '@aller/shiny/src/atoms/Grid'
import { Row } from '@aller/shiny/src/atoms/Row'
import { Col } from '@aller/shiny/src/atoms/Col'
import { VerticalLinkBar } from '@aller/shiny/src/molecules/VerticalLinkBar'
import { LinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'
import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { getColor } from '@aller/shiny/src/utils/get-color'
import withBorder from './withBorder'

// Used div, not a Grid/Row/Col, because if props !fluid it will calculate width by using func `config`
const ColWithBorder = withBorder(['left'])(Col)

// without it VerticalLinkBar doesn't work, because it has a rule display: inline-block; in component VerticalFlexingList: 10
const StyledVerticalLinkBar = styled(VerticalLinkBar)`
	display: flex;
	flex-direction: column;
`

const HoverLinkBarLink = styled(LinkBarLink)`
	&& {
		white-space: normal;
		word-break: break-all;
		letter-spacing: 0rem;
		line-height: 1.6rem;
		&:hover {
			text-decoration: underline;
		}
	}
`
const StyledSmallLinkBarHeading = styled(SmallLinkBarHeading)`
	font-weight: ${getVariable('uiWeightBold')};
	size: ${getVariable('uiSmallSize')};
	line-height: 1.6rem;
	letter-spacing: 0rem;
`
const RedSpan = styled.span`
	color: ${getColor('primary')};
	@media screen and (max-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
		margin-left: 0.4rem;
	}
`
const StyledRow = styled(Row)`
	padding: 1.5rem 0;
`
export const ImportantLabel = () => <RedSpan>*</RedSpan>

const DesktopMenu = ({ data }) => (
	<Grid>
		<StyledRow nowrap>
			{data.map(({ title, data }) => {
				const newData = data.map(t => t)
				newData.unshift({ title })
				return (
					<ColWithBorder xs key={title}>
						<StyledVerticalLinkBar backgroundColor="white">
							{newData.map(({ title, url, color, important }) => {
								return url ? (
									<HoverLinkBarLink
										key={title}
										useUnderline={false}
										size="small"
										textColor={color}
										ALLCAPS
										url={url}
									>
										{title}
										{important && <ImportantLabel />}
									</HoverLinkBarLink>
								) : (
									<StyledSmallLinkBarHeading
										key={title}
										textColor="primary"
										marginBottomFactor={0}
										upSize="Small"
										ALLCAPS
									>
										{title}
									</StyledSmallLinkBarHeading>
								)
							})}
						</StyledVerticalLinkBar>
					</ColWithBorder>
				)
			})}
		</StyledRow>
	</Grid>
)
DesktopMenu.propTypes = {
	data: PropTypes.arrayOf(
		PropTypes.shape({
			title: PropTypes.string.isRequired,
			data: PropTypes.arrayOf(
				PropTypes.shape({
					title: PropTypes.string.isRequired,
					color: PropTypes.string,
					important: PropTypes.bool,
					url: PropTypes.string.isRequired,
				}),
			),
		}),
	).isRequired,
}
export default DesktopMenu
