import React from 'react'
import dynamic from 'next/dynamic'
import propTypes from 'prop-types'
import styled from 'styled-components'

import { getColor } from '@aller/shiny/src/utils/get-color'
import { Standfirst } from '@aller/shiny/src/atoms/Standfirst'
import { BlockLink } from '@aller/shiny/src/atoms/BlockLink'
import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { VerticalFlexingList } from '@aller/shiny/src/atoms/VerticalFlexingList'
import { LargeLinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'
import Frame from './SOLComponents/Frame'

const KKLogo = dynamic(() =>
	import(/* webpackChunkName: "KKLogo" */ '@aller/shiny/src/atoms/KKLogo').then(c => c.KKLogo),
)

const LogoLink = styled(LargeLinkBarLink)`
	& {
		width: 5.5rem;
		padding: 0;
	}
`
const FooterTag = styled.footer`
	border-top: 0.1rem solid ${getColor('grayTintLighter')};
	margin: 0;
	padding: ${getVariable('verticalBase')};
	a {
		border: none;
		text-underline: none;
		color: ${getColor('darkness')};
	}
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		padding: 0;
		padding-top: ${getVariable('verticalBase')};
	}
`

const NavLinksBar = styled(VerticalFlexingList)`
	& {
		display: block;
		list-style: none;
		margin: 0;
		line-height: ${getVariable('headingRegularLineHeight')};
		padding: 0;
	}
`

const BottomParagraph = styled(Standfirst)`
	margin: 2rem 0;
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		margin-top: 2rem;
	}
`

const KKLink = styled(BlockLink)`
	line-height: calc(${getVariable('uiRegularSize')} * 2);
	text-decoration: underline;
	margin-bottom: ${props => (props.mb ? props.mb : 1)}rem;
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		line-height: calc(${getVariable('uiRegularSize')} * 1.5);
	}
`

const Address = styled.address`
	font-style: normal;
	font-weight: ${props => (props.bold ? getVariable('headingsWeight') : 300)};
`

const KKFooter = ({
	address,
	googleMapUrl,
	rulesUrl,
	cookieUrl,
	contactUrl,
	advertiseUrl,
	privacyLink,
	editorEmail,
}) => {
	return (
		<FooterTag>
			<Frame>
				<NavLinksBar>
					<LogoLink url="https://www.kk.no" useUnderline={false}>
						<KKLogo />
					</LogoLink>
				</NavLinksBar>
				<NavLinksBar>
					<Address bold>Postadresse:</Address>
					<Address> 1184 Sentrum, 0107 Oslo</Address>
					<Address bold>Besøksadresse:</Address>
					<KKLink href={googleMapUrl}>{address}</KKLink>
				</NavLinksBar>
				<NavLinksBar>
					<KKLink href={rulesUrl}>Redaksjonelle retningslinjer</KKLink>
					<KKLink href={cookieUrl}>Om informasjonskapsler</KKLink>
					<KKLink href={advertiseUrl}>Annonseinformasjon</KKLink>
					<KKLink href={privacyLink}>Personvernerklæring</KKLink>
				</NavLinksBar>
				<NavLinksBar>
					<b>Ansvarlig redaktør:</b>
					<KKLink href={`mailto:${editorEmail}`} mb={3}>
						Ingeborg Heldal
					</KKLink>
					<Address bold>Utgiverselskap:</Address>
					<KKLink href="http://www.aller.no/">Aller Media AS</KKLink>
				</NavLinksBar>
				<BottomParagraph>
					KK gjør oppmerksom på at alle bloggere på denne nettsiden er uavhengige, med eget
					innholdsansvar.
				</BottomParagraph>
			</Frame>
		</FooterTag>
	)
}

KKFooter.propTypes = {
	address: propTypes.string,
	googleMapUrl: propTypes.string,
	rulesUrl: propTypes.string,
	cookieUrl: propTypes.string,
	contactUrl: propTypes.string,
	advertiseUrl: propTypes.string,
	privacyLink: propTypes.string,
	editorEmail: propTypes.string,
}

KKFooter.defaultProps = {
	address: 'Karvesvingen 1, 0579 Oslo',
	googleMapUrl: 'https://goo.gl/maps/EmpIH',
	rulesUrl: 'https://www.kk.no/annet/redaksjonelle-retningslinjer-for-kkno/67867154',
	cookieUrl: 'https://personvern.aller.no/cookiepolicy.html',
	contactUrl: 'https://www.sol.no/hjelp',
	advertiseUrl: 'http://annonse.aller.no/',
	privacyLink: 'https://personvern.aller.no/personvern.html',
	editorEmail: 'ingeborg.heldal@aller.com',
}

export default KKFooter
