import React, { Component } from 'react'
import PropTypes from 'prop-types'
import styled from 'styled-components'
import { Heading as ShinyHeading } from '@aller/shiny/src/atoms/Heading'
import { getColor } from '@aller/shiny/src/utils/get-color'
import { createSelector } from 'reselect'
import { Row } from '@aller/shiny/src/atoms/Row'
import { Col } from '@aller/shiny/src/atoms/Col'
import { Grid } from '@aller/shiny/src/atoms/Grid'
import { getVariable } from '@aller/shiny/src/utils/get-variable'
import BackgroundScreen from './BackgroundScreen'
import PolygonCanvas from './PolygonCanvas'
import { tagSummarizeSelector } from '../../utils/articleSelectors'
import Slider from './Slider'

const verticesSelector = createSelector(
	[
		({ tagSummarizePercent, sortedKeys }) => {
			return sortedKeys.map(key => tagSummarizePercent[key])
		},
	],
	vertices => ({ vertices }),
)

const Heading = styled(ShinyHeading)`
	color: ${p => getColor(p.color)};
	text-align: center;
`
const List = styled.ul`
	text-align: left;
`
const Li = styled.li`
	margin: calc(${getVariable('verticalBase')} * 2) ${getVariable('horizontalBase')};
`
const prepareState = ({ articles, visible }) => {
	const { sortedKeys, tagSummarizePercent } = tagSummarizeSelector({ articles, visible })
	const { vertices } = verticesSelector({ tagSummarizePercent, sortedKeys })
	return {
		vertices,
		sortedKeys,
		tagSummarizePercent: { ...tagSummarizePercent },
	}
}

class Interests extends Component {
	constructor(props) {
		super(props)
		const { data, visible } = props
		this.state = prepareState({ articles: data.data, visible })
	}

	componentWillReceiveProps({ data, visible }) {
		this.setState(prepareState({ articles: data.data, visible }))
	}

	onChange = (key, value) => {
		const { tagSummarizePercent: oldTagSummarizePercent, sortedKeys } = this.state
		const tagSummarizePercent = { ...oldTagSummarizePercent, [key]: value * 1 }
		const { vertices } = verticesSelector({ tagSummarizePercent, sortedKeys })
		this.setState({ tagSummarizePercent, vertices })
	}

	render() {
		const { vertices, sortedKeys, tagSummarizePercent } = this.state
		return (
			<BackgroundScreen
				center
				backgroundColor="darknessLighter"
				backgroundImage="https://i.pinimg.com/originals/1e/27/6a/1e276aa713753842397527f9ec0efabf.jpg"
				imageColor="darknessLighter"
			>
				<Heading color="white" size="huge">
					Dine interesser
				</Heading>
				<Grid>
					<Row around="xs" middle="xs">
						<Col xs={6}>
							<List>
								{sortedKeys.slice(0, 8).map(text => (
									<Li key={text}>
										<Slider
											label={text}
											onChange={(e, value) => this.onChange(text, value)}
											value={tagSummarizePercent[text]}
										/>
									</Li>
								))}
							</List>
						</Col>
						<Col xs={6}>
							<PolygonCanvas
								vertices={vertices.slice(0, 8)}
								primaryColor="blue"
								secondaryColor="lighter"
							/>
						</Col>
					</Row>
				</Grid>
			</BackgroundScreen>
		)
	}
}

Interests.propTypes = {
	visible: PropTypes.number,
	data: PropTypes.shape({
		data: PropTypes.arrayOf(PropTypes.shape({ tags: PropTypes.arrayOf(PropTypes.string) })),
	}).isRequired,
}
Interests.defaultProps = {
	visible: 8,
}

export default Interests
