import React from 'react'
import stripTags from 'striptags'

import { useIsMobile } from '@aller/shared/hooks/useIsMobile'
import getHostname from '../../../utils/get-hostname'
import fillEmptyGridSizes from './get-grid-sizes'

import { SkistarArticle } from './styles/ArticleCol'
import PlugLink from './styles/PlugLink'
import LabradorLabel from './styles/LabradorLabel'
import ArticlePreviewFooter from './styles/ArticlePreviewFooter'
import Description from './styles/Description'
import Headline from './styles/Headline'
import Source from './styles/Source'
import ArticlePreviewText from './styles/ArticlePreviewText'
import LoftPlugImage from './styles/LoftPlugImage'
import LazyImage from '../../LazyImage'
import { cloudify } from '../../../utils/generate-cloudify-image-urls'

import { ImageSource } from '../../../types/images'
import { Column } from '../../FlexCol/types'
import { generateLabradorImageUrls } from '../../../utils/generate-labrador-image-urls'
import generateCloudifyImageUrls from '../../../utils/generate-cloudify-image-urls'

function calculateRatioByTitle(title: string) {
	const { length } = stripTags(title, ['strong', 'em'])

	switch (true) {
		case length <= 44:
			return 1.05
		case length > 54:
			return 1.15
		case length > 54 && length <= 68:
			return 1.36
		case length > 68 && length <= 100:
			return 1.45
		case length >= 100:
			return 1.6
		default:
			return 1.15
	}
}

function generateImageSources(
	articleSource: string,
	imageUrl: string,
	originalImageUrl: string,
	imageRatio: number,
	gridColumnSizes: Column,
): ImageSource[] | undefined {
	switch (articleSource) {
		case 'Labrador':
		case 'Xavier':
			return generateLabradorImageUrls(imageUrl, imageRatio, gridColumnSizes, true)
			break
		case 'Harvester':
			return generateCloudifyImageUrls(originalImageUrl, imageRatio, gridColumnSizes)
			break
		default:
			return undefined
	}
}

function generateLoftPlugImage(
	articleSource: string,
	imageUrl: string,
	originalImageUrl: string,
	imageRatio: number,
) {
	switch (articleSource) {
		case 'Labrador':
		case 'Xavier':
			imageUrl = `${imageUrl.split('&')[0]}&width=1200&compression=100`
			break
		case 'Harvester':
			imageUrl = originalImageUrl
			break
	}

	return cloudify(imageUrl, {
		width: 100,
		height: Math.round(imageRatio * 100),
		quality: 100,
	})
}

interface SolPlugProps {
	imageUrl?: string
	originalImageUrl?: string
	imageSources?: ImageSource[]
	imageRatio?: number
	loading?: 'eager' | 'auto' | 'lazy'
	preventBlur?: boolean
	column: Column
	fontColumnSizes?: any
	url?: string
	isLoftPlug?: boolean
	useBoxShadow?: boolean
	labelColor?: string
	fontSizeOverride?: number
	subtitle?: string
	title: string
	blinkPreset?: string
	dataLabel?: string
	[x: string]: any
	colorTheme?: string
	linkpulsePersonalizedClass?: boolean
}

const SolPlug: React.FC<SolPlugProps> = ({
	imageUrl = '',
	originalImageUrl = '',
	imageSources,
	imageRatio = 0.5,
	subtitle,
	labelColor,
	title,
	isLoftPlug = false,
	useBoxShadow,
	blinkPreset,
	dataLabel,
	column: gridColumnSizes,
	fontColumnSizes,
	fontSizeOverride = 0,
	url,
	colorTheme = '',
	loading,
	linkpulsePersonalizedClass,
	articleSource,
}) => {
	gridColumnSizes = fillEmptyGridSizes(gridColumnSizes)
	if (fontColumnSizes) {
		fontColumnSizes = fillEmptyGridSizes(fontColumnSizes)
	} else {
		fontColumnSizes = gridColumnSizes
	}

	if (useIsMobile()) {
		if (fontColumnSizes.md < 7) {
			isLoftPlug = true
			imageRatio = calculateRatioByTitle(title)
		} else {
			imageRatio = 0.5
		}
	}

	let loftPlugImage = ''
	if (imageUrl) {
		if (!imageSources) {
			imageSources = generateImageSources(
				articleSource,
				imageUrl,
				originalImageUrl,
				imageRatio,
				gridColumnSizes,
			)
		}

		if (isLoftPlug) {
			loftPlugImage = generateLoftPlugImage(
				articleSource,
				imageUrl,
				originalImageUrl,
				calculateRatioByTitle(title),
			)
		}
	}

	const theClasses = `${linkpulsePersonalizedClass ? ' lp_personalized' : ''}`

	const cxense_attr: any = {}
	if (blinkPreset) {
		cxense_attr['data-cxense_tag'] = 'cerebro'
		cxense_attr['data-cxence_widget'] = blinkPreset
	}

	return (
		<SkistarArticle
			className="preview-blink lp_article"
			{...cxense_attr}
			{...gridColumnSizes}
			colorTheme={colorTheme}
		>
			<PlugLink
				className={theClasses}
				useBoxShadow={useBoxShadow}
				colorTheme={colorTheme}
				target="_blank"
				href={url}
				flexDirectionRow={isLoftPlug}
			>
				{!isLoftPlug && imageUrl && imageSources && imageSources[0] && imageSources[0].url && (
					<div style={{ order: 2 }}>
						<LazyImage
							sources={imageSources}
							alt={stripTags(title)}
							ratio={imageRatio}
							offset={0}
							src={imageUrl}
							preventBlur
							fadeIn={false}
							loading={loading}
						>
							{imageSources.length === 0 && <Source srcSet={imageUrl} />}
							{imageSources.map((source, i) => {
								if (i === 2) {
									return null
								}
								return <Source srcSet={source.url} media={source.media} key={`source-${i}`} />
							})}
						</LazyImage>
					</div>
				)}
				{isLoftPlug && imageUrl && <LoftPlugImage image={loftPlugImage} />}
				<ArticlePreviewText>
					{!!dataLabel && <LabradorLabel labelColor={labelColor}>{dataLabel}</LabradorLabel>}
					<Headline
						order={3}
						fontSizeOverride={fontSizeOverride}
						{...fontColumnSizes}
						dangerouslySetInnerHTML={{ __html: title }} // eslint-disable-line react/no-danger
					/>
					{subtitle && (
						<Description
							colorTheme={colorTheme}
							{...gridColumnSizes}
							order={4}
							itemProp="description"
						>
							{subtitle}
						</Description>
					)}
					<ArticlePreviewFooter colorTheme={colorTheme}>
						<span>{getHostname(url)}</span>
					</ArticlePreviewFooter>
				</ArticlePreviewText>
			</PlugLink>
		</SkistarArticle>
	)
}

export default SolPlug
