import React from 'react'
import PropTypes from 'prop-types'
import stripTags from 'striptags'
import styled from 'styled-components'

import { PlugHeading as DefaultHeading } from '@aller/shiny/src/atoms/PlugHeading'
import { Labels as DefaultLabels } from '@aller/shiny/src/molecules/Labels'

import getHostname from '../../utils/get-hostname'
import { fillEmptyGridSizes } from './get-grid-sizes'

import { SkistarArticle } from './ArticleCol'
import PlugLink from './PlugLink'
import LabradorLabel from './LabradorLabel'
import ArticlePreviewFooter from './ArticlePreviewFooter'
import Description from './Description'
import ArticleKicker from './ArticleKicker'
import Figure from './Figure'
import Headline from './Headline'
import Source from './Source'
import ArticlePreviewText from './ArticlePreviewText'

const TitlePlaceholder = styled.div`
	height: 2.4rem;
	background-color: #edecef;
	width: ${prop => prop.width}%;
	margin-bottom: 7px;
	border-radius: 2px;
`

function getLabelColor(name) {
	return name.replace(/\s/g, '').toLowerCase()
}

const SkiStarPlug = ({
	url,
	image,
	ratio,
	title,
	fadeIn,
	Labels,
	labelsProps,
	labels,
	labradorLabel,
	isBlack,
	isLoading,
	isNativeAdvertisement,
	kicker,
	offset,
	Heading,
	commercialPartner,
	showHeroPreview,
	sources,
	stripHeadingHTML,
	subtitle,
	preventBlur,
	headingProps,
	hideImage,
	maxsize,
	logoURL,
	placeholderUrl,
	column,
	attributes,
	bonusColumn,
	fontSizeOverride,
	ignoreDesktopGridFontsize,
	float,
}) => {
	const tmpImage = `https://sol.no/app/cimg/h_270,w_480,c_fill,g_face/${encodeURIComponent(
		`${sources[2].url.split('?')[0]}?width=1200&compression=100`,
	)}`
	sources[3] = {
		url: tmpImage,
		media: '',
	}

	// If the plug has a parent column that defines it's width, we want them to be 12 but still keep their original font size
	const fontColumn = column
	if (maxsize) {
		column = {
			xs: 12,
			sm: 12,
			md: 12,
			lg: 12,
		}
	}

	return (
		<SkistarArticle
			className="preview-blink lp_article"
			{...column}
			isBlack={isBlack}
			isNativeAdvertisement={isNativeAdvertisement}
		>
			<PlugLink
				isNativeAdvertisement={isNativeAdvertisement}
				isBlack={isBlack}
				target="_blank"
				href={url}
			>
				{kicker && (
					<ArticleKicker
						backgroundHoverShade={isNativeAdvertisement && '#fff'}
						isNativeAdvertisement={isNativeAdvertisement}
						order={1}
					>
						{kicker}
						{logoURL && <img src={logoURL} alt={commercialPartner} />}
					</ArticleKicker>
				)}
				{image && sources[0].url && !hideImage && (
					<div style={{ order: 2 }}>
						<Figure
							alt={stripTags(title)}
							ratio={ratio}
							offset={0}
							src={placeholderUrl ? placeholderUrl : image}
							preventBlur={preventBlur}
							fadeIn={fadeIn}
							coudinaryRatio
						>
							{sources.length === 0 && <Source srcSet={image} />}
							{sources.map((source, i) => {
								if (i === 2) {
									return null
								}
								return <Source srcSet={source.url} media={source.media} key={`source-${i}`} />
							})}
						</Figure>
					</div>
				)}
				<ArticlePreviewText {...fillEmptyGridSizes(column)}>
					{!!labradorLabel && (
						<LabradorLabel labradorLabel={getLabelColor(labradorLabel)}>
							{labradorLabel}
						</LabradorLabel>
					)}
					{isLoading && (
						<div style={{ paddingTop: '10px' }}>
							<TitlePlaceholder width={100} />
							<TitlePlaceholder width={60} />
						</div>
					)}
					{!stripHeadingHTML ? (
						<Headline
							fontSizeOverride={fontSizeOverride}
							{...headingProps}
							order={3}
							ignoreDesktopGridFontsize={ignoreDesktopGridFontsize}
							bonusColumn={bonusColumn}
							maxsize={maxsize}
							{...fillEmptyGridSizes(fontColumn)}
							dangerouslySetInnerHTML={{ __html: title }} // eslint-disable-line react/no-danger
						/>
					) : (
						<Headline
							fontSizeOverride={fontSizeOverride}
							bonusColumn={bonusColumn}
							ignoreDesktopGridFontsize={ignoreDesktopGridFontsize}
							{...fillEmptyGridSizes(fontColumn)}
							{...headingProps}
							maxsize={maxsize}
							order={3}
						>
							{stripTags(title, ['strong', 'em'])}
						</Headline>
					)}
					{subtitle && (
						<Description
							isBlack={isBlack}
							{...fillEmptyGridSizes(column)}
							order={4}
							itemProp="description"
						>
							{subtitle}
						</Description>
					)}
					{!!labels.length && <Labels {...labelsProps} labels={labels} />}
					<ArticlePreviewFooter isBlack={isBlack}>
						{!isNativeAdvertisement && <span>{getHostname(url)}</span>}
					</ArticlePreviewFooter>
				</ArticlePreviewText>
			</PlugLink>
		</SkistarArticle>
	)
}

SkiStarPlug.displayName = 'Plug'

SkiStarPlug.propTypes = {
	/** Extra attributes you want on the DOM-node. e.g. data-cxense_Tag */
	attributes: PropTypes.object, // eslint-disable-line
	/** Option to fadein the image. */
	fadeIn: PropTypes.bool,
	/** Will be passed through to ArticleComponent */
	float: PropTypes.oneOf([
		'left',
		'right',
		'none',
		'inline-start',
		'inline-end',
		'inherit',
		'initial',
		'unset',
	]),
	/** Text above the image plug. */
	kicker: PropTypes.string,
	/** Override the font-size set by the grid  */
	fontSizeOverride: PropTypes.string,
	/** Logo for content ads */
	logoURL: PropTypes.string,
	/** Main link text. */
	title: PropTypes.string,
	/** Should the image be displayed */
	hideImage: PropTypes.bool,
	/** Is this plug inside a top-normal strucutre?  */
	maxsize: PropTypes.bool,
	/** Override the source marking on the plug */
	commercialPartner: PropTypes.string,
	/** Is this a dark/black plug */
	isBlack: PropTypes.bool,
	/** Should we strip the HTML tags from the heading */
	isLoading: PropTypes.bool,
	isNativeAdvertisement: PropTypes.bool,
	/** Should add class */
	stripHeadingHTML: PropTypes.bool,
	/** Norwegian: "ingress" */
	subtitle: PropTypes.string,
	/** Primary source URL for the plug image. If it is empty, no image will be displayed. */
	image: PropTypes.string.isRequired,
	/** Labels added in labrador  */
	labradorLabel: PropTypes.string.isRequired,
	/** Component to display the labels */
	Labels: PropTypes.oneOfType([PropTypes.func, PropTypes.shape({ render: PropTypes.func })]),
	/** Props to pass on to the Heading component */
	labelsProps: PropTypes.shape({
		order: PropTypes.number,
	}),
	/** Do not change font size depending on the grid sizes */
	ignoreDesktopGridFontsize: PropTypes.bool,
	bonusColumn: PropTypes.bool,
	/** i.e. keywords, tags, labels, categories */
	labels: PropTypes.arrayOf(
		PropTypes.shape({
			backgroundColor: PropTypes.string,
			backgroundHoverShade: PropTypes.oneOf(['', 'dark', 'light', 'lighter']),
			textColor: PropTypes.string,
			text: PropTypes.string,
		}),
	),
	/** The user will be sent here if he clicks on the plug. */
	url: PropTypes.string.isRequired,
	/** If given, this image will be displayed before load. If empty, the 'image' prop will be used instead. */
	placeholderUrl: PropTypes.string,
	/** How close can the image be to the viewport before it is lazily loaded? */
	offset: PropTypes.number,
	/** Set of images to be rendered next to placeholderUrl and image */
	sources: PropTypes.arrayOf(PropTypes.object),
	/** Width of the image divided by height of the image */
	ratio: PropTypes.number.isRequired,
	/** Component to display the title */
	Heading: PropTypes.oneOfType([PropTypes.func, PropTypes.shape({ render: PropTypes.func })]),
	/** Column object to decide wether the article should be a Col or not */
	column: PropTypes.shape({
		width: PropTypes.number,
		reverse: PropTypes.bool,
		xs: PropTypes.number,
		sm: PropTypes.number,
		md: PropTypes.number,
		lg: PropTypes.number,
		xsOffset: PropTypes.number,
		smOffset: PropTypes.number,
		mdOffset: PropTypes.number,
		lgOffset: PropTypes.number,
		children: PropTypes.node,
	}),
	showHeroPreview: PropTypes.bool,
	/** Props to pass on to the Heading component */
	headingProps: PropTypes.shape({
		skin: PropTypes.shape({
			backgroundColor: PropTypes.string,
			textColor: PropTypes.string,
			needsPadding: PropTypes.bool,
		}),
	}),
	/** Disable the blur effect on lazily loaded images */
	preventBlur: PropTypes.bool,
	displayPlayIcon: PropTypes.bool,
	playIconColor: PropTypes.string,
}
SkiStarPlug.defaultProps = {
	attributes: {},
	column: null,
	fadeIn: false,
	float: 'initial',
	logoURL: '',
	maxsize: false,
	hideImage: false,
	kicker: '',
	title: '',
	ignoreDesktopGridFontsize: false,
	isBlack: false,
	bonusColumn: false,
	fontSizeOverride: 0,
	isLoading: false,
	isNativeAdvertisement: false,
	subtitle: '',
	commercialPartner: '',
	Labels: DefaultLabels,
	stripHeadingHTML: true,
	labelsProps: {
		order: 5,
	},
	labels: [],
	showHeroPreview: false,
	offset: 0,
	sources: [],
	Heading: DefaultHeading,
	headingProps: {},
	preventBlur: false,
	placeholderUrl: '',
	displayPlayIcon: false,
	playIconColor: 'primary',
}

export default SkiStarPlug
