import React from 'react'
import PropTypes from 'prop-types'
import stripTags from 'striptags'
import styled from 'styled-components'

import { PlugHeading as DefaultHeading } from '@aller/shiny/src/atoms/PlugHeading'
import { Labels as DefaultLabels } from '@aller/shiny/src/molecules/Labels'

import getHostname from '../../utils/get-hostname'
import { fillEmptyGridSizes } from './get-grid-sizes'

import { SmallSkiStarArticle } from './ArticleCol'
import PlugLink from './PlugLink'
import LabradorLabel from './LabradorLabel'
import ArticlePreviewFooter from './ArticlePreviewFooter'
import Description from './Description'
import ArticleKicker from './ArticleKicker'
import Headline from './Headline'
import ArticlePreviewText from './ArticlePreviewText'

const TitlePlaceholder = styled.div`
	height: 2.4rem;
	background-color: #edecef;
	width: ${prop => prop.width}%;
	margin-bottom: 7px;
	border-radius: 2px;
`

function getLabelColor(name) {
	return name.replace(/\s/g, '').toLowerCase()
}

const SmallPreview = ({
	url,
	image,
	ratio,
	title,
	fadeIn,
	Labels,
	labelsProps,
	labels,
	labradorLabel,
	isBlack,
	hideImage,
	isLoading,
	isNativeAdvertisement,
	kicker,
	offset,
	Heading,
	commercialPartner,
	showHeroPreview,
	sources,
	stripHeadingHTML,
	subtitle,
	preventBlur,
	headingProps,
	logoURL,
	placeholderUrl,
	column,
	attributes,
	maxsize,
	bonusColumn,
	fontSizeOverride,
	ignoreDesktopGridFontsize,
	float,
}) => {
	const tmpImage = sources[2].url.split('?')
	const mobileImage = `https://sol.no/app/cimg/h_${
		stripTags(title, ['strong', 'em']).length < 55 ? 115 : 136
	},w_100,c_fill,g_faces/${encodeURIComponent(`${tmpImage[0]}?width=1200&compression=100`)}`

	const CurrentImage = styled.div`
		order: 3;
		flex-direction: row-reverse;
		flex-shrink: 0;
		background-position: top center;
		background-repeat: no-repeat;
		background-image: url("${mobileImage}");
		width: 100px!important;
		min-height: 100px;
		height: 100%;
		max-height: 136px;
	`

	// If the plug has a parent column that defines it's width, we want them to be 12 but still keep their original font size
	const fontColumn = column
	if (maxsize) {
		column = {
			xs: 12,
			sm: 12,
			md: 12,
			lg: 12,
		}
	}

	return (
		<SmallSkiStarArticle
			className="preview-blink lp_article"
			{...column}
			isBlack={isBlack}
			isNativeAdvertisement={isNativeAdvertisement}
		>
			<PlugLink
				isNativeAdvertisement={isNativeAdvertisement}
				isBlack={isBlack}
				target="_blank"
				href={url}
				flexDirectionRow
			>
				{kicker && (
					<ArticleKicker
						backgroundHoverShade={isNativeAdvertisement && '#fff'}
						isNativeAdvertisement={isNativeAdvertisement}
						order={1}
					>
						{kicker}
						{logoURL && <img src={logoURL} alt={commercialPartner} />}
					</ArticleKicker>
				)}

				<ArticlePreviewText>
					{!!labradorLabel && (
						<LabradorLabel labradorLabel={getLabelColor(labradorLabel)}>
							{labradorLabel}
						</LabradorLabel>
					)}
					{isLoading && (
						<div style={{ paddingTop: '10px' }}>
							<TitlePlaceholder width={100} />
							<TitlePlaceholder width={60} />
						</div>
					)}
					{!stripHeadingHTML ? (
						<Headline
							fontSizeOverride="2"
							{...headingProps}
							order={2}
							ignoreDesktopGridFontsize={ignoreDesktopGridFontsize}
							bonusColumn={bonusColumn}
							{...fillEmptyGridSizes(fontColumn)}
							dangerouslySetInnerHTML={{ __html: title }} // eslint-disable-line react/no-danger
						/>
					) : (
						<Headline
							fontSizeOverride="2"
							bonusColumn={bonusColumn}
							ignoreDesktopGridFontsize={ignoreDesktopGridFontsize}
							{...fillEmptyGridSizes(fontColumn)}
							{...headingProps}
							order={2}
						>
							{stripTags(title, ['strong', 'em'])}
						</Headline>
					)}
					{subtitle && (
						<Description {...fillEmptyGridSizes(column)} order={4} itemProp="description">
							{subtitle}
						</Description>
					)}
					{!!labels.length && <Labels {...labelsProps} labels={labels} />}
					<ArticlePreviewFooter isBlack={isBlack}>
						{!isNativeAdvertisement && <span>{getHostname(url)}</span>}
					</ArticlePreviewFooter>
				</ArticlePreviewText>
				{image && sources[2].url && !hideImage && <CurrentImage />}
			</PlugLink>
		</SmallSkiStarArticle>
	)
}

SmallPreview.displayName = 'Plug'

SmallPreview.propTypes = {
	/** Extra attributes you want on the DOM-node. e.g. data-cxense_Tag */
	attributes: PropTypes.object, // eslint-disable-line
	/** Option to fadein the image. */
	fadeIn: PropTypes.bool,
	/** Will be passed through to ArticleComponent */
	float: PropTypes.oneOf([
		'left',
		'right',
		'none',
		'inline-start',
		'inline-end',
		'inherit',
		'initial',
		'unset',
	]),
	/** Text above the image plug. */
	kicker: PropTypes.string,
	/** Should the image be hidden  */
	hideImage: PropTypes.bool,
	/** Override the font-size set by the grid  */
	fontSizeOverride: PropTypes.string,
	/** Logo for content ads */
	logoURL: PropTypes.string,
	/** Main link text. */
	title: PropTypes.string,
	/** Override the source marking on the plug */
	commercialPartner: PropTypes.string,
	/** Is this a dark/black plug */
	isBlack: PropTypes.bool,
	/** Is this plug inside a top-normal strucutre?  */
	maxsize: PropTypes.bool,
	/** Should we strip the HTML tags from the heading */
	isLoading: PropTypes.bool,
	isNativeAdvertisement: PropTypes.bool,
	/** Should add class */
	stripHeadingHTML: PropTypes.bool,
	/** Norwegian: "ingress" */
	subtitle: PropTypes.string,
	/** Primary source URL for the plug image. If it is empty, no image will be displayed. */
	image: PropTypes.string.isRequired,
	/** Labels added in labrador  */
	labradorLabel: PropTypes.string.isRequired,
	/** Component to display the labels */
	Labels: PropTypes.oneOfType([PropTypes.func, PropTypes.shape({ render: PropTypes.func })]),
	/** Props to pass on to the Heading component */
	labelsProps: PropTypes.shape({
		order: PropTypes.number,
	}),
	/** Do not change font size depending on the grid sizes */
	ignoreDesktopGridFontsize: PropTypes.bool,
	bonusColumn: PropTypes.bool,
	/** i.e. keywords, tags, labels, categories */
	labels: PropTypes.arrayOf(
		PropTypes.shape({
			backgroundColor: PropTypes.string,
			backgroundHoverShade: PropTypes.oneOf(['', 'dark', 'light', 'lighter']),
			textColor: PropTypes.string,
			text: PropTypes.string,
		}),
	),
	/** The user will be sent here if he clicks on the plug. */
	url: PropTypes.string.isRequired,
	/** If given, this image will be displayed before load. If empty, the 'image' prop will be used instead. */
	placeholderUrl: PropTypes.string,
	/** How close can the image be to the viewport before it is lazily loaded? */
	offset: PropTypes.number,
	/** Set of images to be rendered next to placeholderUrl and image */
	sources: PropTypes.arrayOf(PropTypes.object),
	/** Width of the image divided by height of the image */
	ratio: PropTypes.number.isRequired,
	/** Component to display the title */
	Heading: PropTypes.oneOfType([PropTypes.func, PropTypes.shape({ render: PropTypes.func })]),
	/** Column object to decide wether the article should be a Col or not */
	column: PropTypes.shape({
		width: PropTypes.number,
		reverse: PropTypes.bool,
		xs: PropTypes.number,
		sm: PropTypes.number,
		md: PropTypes.number,
		lg: PropTypes.number,
		xsOffset: PropTypes.number,
		smOffset: PropTypes.number,
		mdOffset: PropTypes.number,
		lgOffset: PropTypes.number,
		children: PropTypes.node,
	}),
	showHeroPreview: PropTypes.bool,
	/** Props to pass on to the Heading component */
	headingProps: PropTypes.shape({
		skin: PropTypes.shape({
			backgroundColor: PropTypes.string,
			textColor: PropTypes.string,
			needsPadding: PropTypes.bool,
		}),
	}),
	/** Disable the blur effect on lazily loaded images */
	preventBlur: PropTypes.bool,
	displayPlayIcon: PropTypes.bool,
	playIconColor: PropTypes.string,
}
SmallPreview.defaultProps = {
	attributes: {},
	column: null,
	fadeIn: false,
	hideImage: false,
	float: 'initial',
	logoURL: '',
	kicker: '',
	title: '',
	ignoreDesktopGridFontsize: false,
	isBlack: false,
	bonusColumn: false,
	maxsize: false,
	fontSizeOverride: 0,
	isLoading: false,
	isNativeAdvertisement: false,
	subtitle: '',
	commercialPartner: '',
	Labels: DefaultLabels,
	stripHeadingHTML: true,
	labelsProps: {
		order: 5,
	},
	labels: [],
	showHeroPreview: false,
	offset: 0,
	sources: [],
	Heading: DefaultHeading,
	headingProps: {},
	preventBlur: false,
	placeholderUrl: '',
	displayPlayIcon: false,
	playIconColor: 'primary',
}

export default SmallPreview
