import React, { useState, useEffect } from 'react'
import styled, { withTheme } from 'styled-components'
import PropTypes from 'prop-types'
import { getColor } from '@aller/shiny/src/utils/get-color'
import { getVariable } from '@aller/shiny/src/utils/get-variable'

const Box = styled.div`
	box-sizing: border-box;
	position: fixed;
	bottom: 0;
	left: 0;
	right: 0;
	margin: 0 auto;
	width: 100%;
	max-width: ${props => props.theme.flexboxgrid.container.lg + 1.5}rem;
	min-height: 15rem;
	border-top-left-radius: 1.7rem;
	border-top-right-radius: 1.7rem;
	box-shadow: 0 0 1.4rem 0 rgba(0, 0, 0, 0.25);
	background-color: ${props => {
		const bg = getColor('background')(props)
		if (bg === 'transparent') {
			return '#fff'
		}
		return bg
	}};
	z-index: 5;
	padding: 2.6rem 7rem;
	font-family: ${props => getVariable('mainFont')};

	@media screen and (max-width: ${props => props.theme.flexboxgrid.breakpoints.sm}em) {
		padding: 1.5rem 2.9rem;
	}
`

const Title = styled.h1`
	font-size: ${props => getVariable('headingRegularSize')};
	line-height: ${props => getVariable('headingRegularLineHeight')};
	font-weight: ${props => getVariable('uiWeightBold')};
	color: ${props => getColor('type')};
	margin-bottom: 0.5rem;

	@media screen and (max-width: ${props => props.theme.flexboxgrid.breakpoints.sm}em) {
		font-size: ${props => getVariable('headingMediumSize')};
		line-height: ${props => getVariable('headingMediumLineHeight')};
	}
`

const Body = styled.p`
	color: ${props => getColor('type')};
	@media screen and (max-width: ${props => props.theme.flexboxgrid.breakpoints.sm}em) {
		font-size: ${props => getVariable('uiSmallSize')};
		line-height: ${props => getVariable('uiSmallLineHeight')};
	}
`

const ReadMoreButton = styled.a`
	cursor: pointer;
	padding: 0;
	text-decoration: none;
	display: inline-block;
	text-align: center;
	vertical-align: middle;
	font-size: ${props => getVariable('uiRegularSize')};
	width: 20rem;
	height: 4rem;
	line-height: 4rem;
	border-radius: 1.8rem;
	border: solid 0.1rem ${props => getColor('grayTint')};
	background-color: ${props => getColor('background')};
	color: ${props => getColor('type')};
	margin-right: 2rem;
	font-weight: ${props => getVariable('uiWeightBold')};

	&:hover {
		text-decoration: underline;
	}

	@media screen and (max-width: ${props => props.theme.flexboxgrid.breakpoints.sm}em) {
		font-size: ${props => getVariable('uiSmallSize')};
		line-height: 3.2rem;
		height: 3.2rem;
		width: 11.5rem;
		margin-right: 0.5rem;
	}
`

const ReadMoreButtonButton = ReadMoreButton.withComponent('button')
const OkayButton = styled(ReadMoreButtonButton)`
	background-color: ${props => props.theme.colors.primary};
	border: solid 0.1rem ${props => props.theme.colors.primary};
	color: #fff;
`

const localStorageKey = 'pp_notified'
const PrivacyPopup = ({ site, theme: { name } }) => {
	const [show, setShow] = useState(false)

	function handleClick() {
		setShow(false)
		try {
			localStorage.setItem(localStorageKey, `${Date.now()}`)
		} catch (error) {
			console.log(error) // eslint-disable-line
		}
	}

	useEffect(() => {
		if (typeof localStorage !== 'undefined') {
			try {
				const notified = localStorage.getItem(localStorageKey)

				if (!notified) {
					setShow(true)
				}
			} catch (error) {
				console.log(error) // eslint-disable-line
			}
		}
	}, [])

	if (!show) return null
	// Some themes have e.g. "Dark" which is not relevant, use first word
	const themeSite = name.split(' ')[0]
	return (
		<Box>
			<Title>Vi bryr oss om ditt personvern</Title>
			<Body>
				{site || themeSite || 'Dette nettstedet'} er en del av Aller Media, som er ansvarlig for
				dine data. Vi bruker dataene til å forbedre og tilpasse tjenestene, tilbudene og annonsene
				våre.
			</Body>
			<ReadMoreButton href="https://personvern.aller.no/personvern" target="_blank" rel="noopener">
				Les mer
			</ReadMoreButton>
			<OkayButton onClick={handleClick}>Jeg forstår</OkayButton>
		</Box>
	)
}

PrivacyPopup.propTypes = {
	site: PropTypes.string,
	theme: PropTypes.shape({
		name: PropTypes.string,
	}),
}

PrivacyPopup.defaultProps = {
	site: null,
	theme: {
		name: null,
	},
}

export default withTheme(PrivacyPopup)
