import React from 'react'
import PropTypes from 'prop-types'
import styled from 'styled-components'

import { Heading as ShinyHeading } from '@aller/shiny/src/atoms/Heading'
import { getColor } from '@aller/shiny/src/utils/get-color'
import Screen from '../Screen'

const Meter = styled.meter`
	width: 100%;
	height: 5vh;
	max-width: 100rem;
	margin: 0 auto 2.4rem;
`
const MeterWithDefaults = ({ value, ...rest }) => {
	const low = Math.ceil(value / 20)
	const high = Math.ceil(value * 2)
	const optimum = Math.ceil(value / 15)

	return <Meter min="0" low={low} high={high} optimum={optimum} value={value} {...rest} />
}
MeterWithDefaults.propTypes = {
	value: PropTypes.number.isRequired,
}

const Heading = styled(ShinyHeading)`
	color: ${p => getColor(p.color)};
`

const SingleStat = ({ title, ...rest }) => (
	<>
		<Heading color="white">{title}</Heading>
		<MeterWithDefaults {...rest}>{title}</MeterWithDefaults>
	</>
)
SingleStat.propTypes = {
	title: PropTypes.string.isRequired,
}

const sectionToStat = (section, i) => (
	<SingleStat
		max={section.published}
		value={section.read}
		title={`${section.read} out of ${section.published} articles in the ${section.title} section`}
		key={i}
	/>
)

const tagToStat = (section, i) => (
	<SingleStat
		max={section.published}
		value={section.read}
		title={`${section.read} out of ${section.published} articles tagged ${section.title}`}
		key={i}
	/>
)

const PercentagesOfPublished = ({ data }) => {
	const { sections, tags } = data

	return (
		<Screen color="burgundy" center>
			<Heading color="white" size="huge">
				You have read
			</Heading>
			<SingleStat
				max={data.total.published}
				value={data.total.read}
				title={`9 out of all the 146 articles published on ${data.site}`}
			/>
			{sections.map(sectionToStat)}
			{tags.map(tagToStat)}
		</Screen>
	)
}
PercentagesOfPublished.propTypes = {
	data: PropTypes.shape({
		total: PropTypes.shape({
			published: PropTypes.number.isRequired,
			read: PropTypes.number.isRequired,
		}).isRequired,
		sections: PropTypes.arrayOf(
			PropTypes.shape({
				published: PropTypes.number.isRequired,
				read: PropTypes.number.isRequired,
			}),
		).isRequired,
		site: PropTypes.string.isRequired,
		tags: PropTypes.arrayOf(
			PropTypes.shape({
				published: PropTypes.number.isRequired,
				read: PropTypes.number.isRequired,
			}),
		).isRequired,
	}).isRequired,
}

export default PercentagesOfPublished
