import React from 'react'
import propTypes from 'prop-types'
import { connect } from 'react-redux'
import { bindActionCreators } from 'redux'
import { HorizontalLinkBar } from '@aller/shiny/src/molecules/HorizontalLinkBar'
import { getColor, getVariable } from '@aller/shiny/src/utils'
import { withTheme } from 'styled-components'
import NewsPaperButton from './NewsPaperButton'
import NewsPaperAllButton from './NewsPaperAllButton'

import { setActiveNewsPaperInCategory } from '../actions'

class NewsPaperButtons extends React.Component {
	findCategoryDataAndStoreInLocalState = () => {
		const { categorySet, currentCategory } = this.props
		const category = categorySet.find(cat => cat.name === currentCategory)
		this.state = { category }
	}

	findHosts = () => {
		const { category } = this.state
		if (!category.includeHosts) {
			return null
		}
		return category.includeHosts.split(',')
	}

	isActive = hostName => {
		const { category } = this.state
		return category.onlyOneHost === hostName
	}

	makeButton = (hostName, index) => {
		const { settings: { ua: { isDesktop = false } = {} } = {} } = this.context

		const {
			actions: { setActiveNewsPaperInCategory },
			currentCategory
		} = this.props

		return (
			<NewsPaperButton
				key={index}
				setActiveNewsPaperInCategory={setActiveNewsPaperInCategory}
				isActive={this.isActive(hostName)}
				isDesktop={isDesktop}
				hostName={hostName}
				currentCategory={currentCategory}
			/>
		)
	}

	makeAllButton = () => {
		const {
			actions: { setActiveNewsPaperInCategory },
			currentCategory
		} = this.props

		return (
			<NewsPaperAllButton
				key="allButton"
				setActiveNewsPaperInCategory={setActiveNewsPaperInCategory}
				isActive={this.isActive('')}
				currentCategory={currentCategory}
			/>
		)
	}

	render() {
		const { displaySettings } = this.props
		if (displaySettings) {
			return null
		}

		this.findCategoryDataAndStoreInLocalState()
		const hosts = this.findHosts()
		if (!hosts) {
			return null
		}

		const newsPapers = hosts.map(this.makeButton)
		const allButton = this.makeAllButton()
		const newsPaperMenu = [allButton].concat(newsPapers)

		return (
			<HorizontalLinkBar
				shouldHavePadding={false}
				isTopLevelComponent={false}
				width="100%"
				background={`linear-gradient(0deg, ${getColor('grayTint')(this.props)}, transparent .1rem)`}
			>
				{newsPaperMenu}
			</HorizontalLinkBar>
		)
	}
}

NewsPaperButtons.contextTypes = {
	settings: propTypes.shape({
		isDesktop: propTypes.bool
	})
}

NewsPaperButtons.propTypes = {
	actions: propTypes.shape({
		setActiveNewsPaperInCategory: propTypes.func.isRequired
	}).isRequired,
	categorySet: propTypes.arrayOf(propTypes.object).isRequired,
	currentCategory: propTypes.string.isRequired,
	displaySettings: propTypes.bool.isRequired
}

function mapStateToProps(state) {
	return {
		categorySet: state.categorySet,
		currentCategory: state.currentCategory,
		displaySettings: state.displaySettings
	}
}

function mapDispatchToProps(dispatch) {
	return {
		actions: bindActionCreators(
			{
				setActiveNewsPaperInCategory
			},
			dispatch
		)
	}
}

export default withTheme(
	connect(
		mapStateToProps,
		mapDispatchToProps
	)(NewsPaperButtons)
)
