import React from 'react'
import propTypes from 'prop-types'
import { connect } from 'react-redux'
import { bindActionCreators } from 'redux'

import classnames from 'classnames'
import getOpenerRel from '../../../utils/get-opener-rel'

// Components
import ListPlugWithTime from './ListPlugWithTime'

// Module specific stuff
import { setArticleCardData } from '../actions'
import PlugList from '../styles/article-list'

class ArticleList extends React.Component {
	handleHover = index => {
		const { settings: { ua: { isDesktop = true } = {} } = {} } = this.context
		const {
			actions: { setArticleCardData }
		} = this.props

		if (isDesktop) {
			setArticleCardData(index)
		}
	}

	mapArticles = (articles, pages, articlesPerPage, currentCategory) => {
		const articlesToDisplay = articles.slice(
			articlesPerPage * pages - articlesPerPage,
			articlesPerPage * pages
		)
		return articlesToDisplay.map((article, index) => {
			const { host, posted, title, url } = article

			const target = '_blank'
			const openerRel = getOpenerRel(target)
			const rel = classnames('bookmark', openerRel)
			const lpPosition = `newsstream_stream_${currentCategory}_${index + 1}`

			const hoverIndex = articlesPerPage * (pages - 1) + index
			return (
				<ListPlugWithTime
					handleHover={this.handleHover}
					host={host}
					index={hoverIndex}
					isLast={index === articlesPerPage * pages - 1}
					key={index}
					rel={rel}
					target={target}
					title={title}
					updateTime={posted}
					url={url}
					position={lpPosition}
				/>
			)
		})
	}

	renderArticles() {
		const { settings: { ua: { isDesktop = false, isTablet = false } = {} } = {} } = this.context

		const {
			articles,
			currentCategory,
			pagination: {
				articlesPerPage: { mobile, desktop },
				pages
			}
		} = this.props

		if (articles.length < 1) {
			return null
		}

		const articlesPerPage = isDesktop || isTablet ? desktop : mobile

		return this.mapArticles(articles, pages, articlesPerPage, currentCategory)
	}

	render() {
		return <PlugList>{this.renderArticles()}</PlugList>
	}
}

ArticleList.contextTypes = {
	settings: propTypes.shape({
		ua: propTypes.shape({
			isMobile: propTypes.bool,
			isTablet: propTypes.bool,
			isDesktop: propTypes.bool
		})
	})
}

ArticleList.propTypes = {
	actions: propTypes.shape({
		setArticleCardData: propTypes.func.isRequired
	}).isRequired,
	articles: propTypes.arrayOf(propTypes.object).isRequired,
	currentCategory: propTypes.string.isRequired,
	pagination: propTypes.shape({
		pages: propTypes.number.isRequired,
		articlesPerPage: propTypes.shape({
			mobile: propTypes.number.isRequired,
			desktop: propTypes.number.isRequired
		}).isRequired
	}).isRequired
}

function mapDispatchToProps(dispatch) {
	return {
		actions: bindActionCreators(
			{
				setArticleCardData
			},
			dispatch
		)
	}
}

function mapStateToProps(state) {
	return {
		currentCategory: state.currentCategory,
		articles: state.articles,
		pagination: state.pagination
	}
}

export default connect(
	mapStateToProps,
	mapDispatchToProps
)(ArticleList)
