import { Component } from 'react'
import propTypes from 'prop-types'
import qs from 'query-string'

import addListener from './add-listener'
import checkSessionStorage from './check-session-storage'
import doMetaRefresh from './do-meta-refresh'

export default class MetaRefresh extends Component {
	constructor(props) {
		super(props)
		this.resetTimer = this.resetTimer.bind(this)
	}

	componentDidMount() {
		const params = qs.parse(window.location.search)
		if (Object.keys(params).length !== 0 || params.meta) {
			if (params.meta) {
				delete params.meta
			}

			if (Object.keys(params).length !== 0) {
				window.history.pushState({ ...params, hasParams: true }, '', `?${qs.stringify(params)}`)
			} else {
				window.history.pushState('', '', '/')
			}
		}
		const shouldMetaRefresh = window.innerWidth > 1024 && checkSessionStorage
		const { resetOnActivity, metaRefreshInterval } = this.props
		if (shouldMetaRefresh) {
			if (resetOnActivity) {
				const throttle = 1000 // One second
				addListener('mousedown', this.resetTimer, throttle)
				addListener('keydown', this.resetTimer, throttle)
				addListener('scroll', this.resetTimer, throttle)
				addListener('mousemove', this.resetTimer, throttle)
			}
			this.metaRefreshTimer = setTimeout(this.executeMetaRefresh, metaRefreshInterval)
		}
	}

	executeMetaRefresh = () => {
		this.shouldMetaRedirect()
			.then(doMetaRefresh)
			.catch(this.resetTimer)
	}

	shouldMetaRedirect = () => {
		return new Promise((resolve, reject) => {
			const xhr = new XMLHttpRequest()
			xhr.open('HEAD', document.location)
			xhr.onload = () => {
				if (xhr.status >= 200 && xhr.status < 300) {
					return resolve()
				}
				return reject()
			}
			xhr.onerror = () => {
				return reject()
			}
			xhr.send()
		})
	}

	resetTimer() {
		const { metaRefreshInterval } = this.props
		clearTimeout(this.metaRefreshTimer)
		this.metaRefreshTimer = setTimeout(this.executeMetaRefresh, metaRefreshInterval)
	}

	render() {
		return null
	}
}

MetaRefresh.propTypes = {
	resetOnActivity: propTypes.bool,
	metaRefreshInterval: propTypes.number,
}
MetaRefresh.defaultProps = {
	resetOnActivity: false,
	metaRefreshInterval: 1000 * 60 * 10,
}
