import React, { useEffect, useState, useReducer, useCallback, memo } from 'react'
import PropTypes from 'prop-types'
import ulog from 'ulog'
import { FullscreenAd } from '@aller/shiny/src/molecules/FullscreenAd'
import { useInView } from 'react-intersection-observer'
import styled from 'styled-components'
import { useIsMetarefresh } from '../../hooks/useIsMetarefresh'
import { AdSettingsCog } from './AdSettingsCog'

import { useAdsDispatch, useAdsStore, gptSlots } from './Provider'
import { getPrebidConfigForSlot, getSlotNameForPath, getSite, getPath } from './config'

const log = ulog('ads')

const slot = 'ad-fullscreen'
const site = getSite()
const path = getPath()
const prebid = getPrebidConfigForSlot(site, slot)

const StyledFullscreenAd = styled(FullscreenAd)`
	width: ${p => p.width};
	height: ${p => p.height};
`

const Adunit = styled.div`
	div[id^='google_ads_iframe'] {
		width: 100% !important;
		height: ${p => p.height} !important;
	}
`
const defaultState = {
	width: '100%',
	height: '100vh',
	isBannerAdInFullscreenSlot: false,
}
const reducer = (state, action) => {
	if (
		action.width !== state.width ||
		action.height !== state.height ||
		action.isBannerAdInFullscreenSlot !== state.isBannerAdInFullscreenSlot
	) {
		return { ...state, ...action }
	}
	return state
}
const AdSlot = memo(({ dispatch, id, height }) => {
	const handleClick = useCallback(() => dispatch({ type: 'SHOW_MODAL' }), [dispatch])
	return (
		<>
			<AdSettingsCog onClick={handleClick} />
			<Adunit id={`${id}`} height={height} className="adunit wolverine-ads" key={`adslot-${id}`} />
		</>
	)
})
AdSlot.propTypes = {
	dispatch: PropTypes.func.isRequired,
	id: PropTypes.string.isRequired,
	height: PropTypes.string.isRequired,
}

const Fullscreen = ({ pos }) => {
	const dispatch = useAdsDispatch()
	const { arcAds } = useAdsStore()
	const [isIE11, setIsIE11] = useState(false)
	const [shouldHide, setShouldHide] = useState(false)
	const [size, sizeDispatch] = useReducer(reducer, defaultState)

	const id = `${slot}${pos}`

	const isMetaRefresh = useIsMetarefresh()
	const slotName = getSlotNameForPath(site, path, isMetaRefresh)
	const options = {
		slotName,
		dimensions: [
			[[1, 1], [1, 2], [980, 600], [980, 300]],
			[[320, 250], [300, 250], [320, 400], [1, 1], [1, 2]],
		],
		sizemap: {
			breakpoints: [[1000, 600], [0, 0]],
			refresh: true,
		},
		targeting: {
			platform: 'wolverine-logan',
			pos: `fullscreen${pos}`,
		},
		bidding: {
			prebid,
		},
	}

	const [ref, inView] = useInView({
		threshold: 0,
		triggerOnce: true,
		rootMargin: '400px',
	})

	useEffect(() => {
		if (
			navigator.userAgent.indexOf('MSIE') !== -1 ||
			navigator.appVersion.indexOf('Trident/') > 0
		) {
			setIsIE11(true)
		}

		if (window.googletag && window.googletag.cmd) {
			window.googletag.cmd.push(() =>
				window.googletag.pubads().addEventListener('slotRenderEnded', event => {
					const slotId = event.slot.getSlotElementId()
					if (slotId !== id) return
					if (event.isEmpty) {
						setShouldHide(true)
						return
					}

					const [width, height] = event.size

					if (width > 10 && width < 1000) {
						// we have a fake fullscreen, and should
						// set a flag to handle it in the visual
						// component
						sizeDispatch({
							width: `${width}px`,
							height: `${height}px`,
							isBannerAdInFullscreenSlot: true,
						})
					}
				}),
			)
		}

		if (inView && arcAds) {
			dispatch({
				type: 'REGISTER_AD',
				options: {
					id,
					...options,
				},
			})
		}

		return function cleanup() {
			if (gptSlots[id]) {
				log.info(`Cleaning up ad ${id}`)
				window.googletag.destroySlots([gptSlots[id]])
			}
		}
	}, [inView, arcAds]) // eslint-disable-line

	// [[[0,0],[[320,250],[300,250],[320,400],[1,1],[1,2]]],[[1000, 600],[[980,300],[970,250],[980,600],[1,1],[1,2]]]]

	return (
		<StyledFullscreenAd
			height={size.height}
			width={size.width}
			isBannerAdInFullscreenSlot={size.isBannerAdInFullscreenSlot}
			shouldHide={shouldHide}
			isIE11={isIE11}
			ref={ref}
		>
			<AdSlot dispatch={dispatch} id={id} height={size.height} />
		</StyledFullscreenAd>
	)
}

Fullscreen.propTypes = {
	pos: PropTypes.number,
}

Fullscreen.defaultProps = {
	pos: 1,
}
Fullscreen.whyDidYouRender = true
Fullscreen.slot = slot

Fullscreen.whyDidYouRender = true

export default Fullscreen
