import React from 'react'
import styled from 'styled-components'
import PropTypes from 'prop-types'
import { formatDistance } from 'date-fns'
import { nb } from 'date-fns/locale'

import { Heading as ShinyHeading } from '@aller/shiny/src/atoms/Heading'
import { getColor } from '@aller/shiny/src/utils/get-color'
import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { tagSummarizeSelector } from '../../utils/articleSelectors'

import ArticlesReadSection from './ArticlesReadSection'

const Heading = styled(ShinyHeading)`
	font-size: ${getVariable('headingLargeSize')};
	color: ${p => getColor(p.color)};
	margin: 0;
	line-height: 6rem;
	text-align: center;
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		font-size: calc(${getVariable('headingLargeSize')} * 2);
	}
`
const Texting = styled.div`
	color: white;
	display: flex;
	flex-direction: column;
	padding: 0;
	justify-content: center;
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		padding: 5rem;
	}
`
const NewsReaderImage = styled.div`
	flex: 1 0 32%;
	height: 45vh;
	align-self: flex-end;
	background: url('https://i.ibb.co/mcLSPz2/imageedit-4853813856.png') center no-repeat;
	background-size: contain;
	margin-bottom: -${getVariable('verticalBase')};
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		height: 90vh;
	}
`
const HighlightedText = styled.span`
	color: ${getColor('pink')};
	font-size: calc(${getVariable('headingLargeSize')} * 2);
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		font-size: calc(${getVariable('headingLargeSize')} * 3.4);
	}
`
const Main = props => {
	const { statisticType, userData } = props

	const articles = userData.data
	const articlesNumber = articles ? articles.length : 0

	const reducer = (accumulator, item) => {
		return accumulator + item.activeTime
	}

	const readingTime = articles.reduce(reducer, 0)

	const timeSpent = readingTime ? formatDistance(readingTime * 1000, 0, { locale: nb }) : 0
	const tagsNumber = articles ? tagSummarizeSelector({ articles }).sortedKeys.length : 0
	const startingText = statisticType === 'daily' ? 'I dag har du lest' : 'Du leser denne uken'

	return (
		<ArticlesReadSection center>
			<Texting>
				<Heading color="white" size="huge">
					{startingText}
					<HighlightedText>{` ${articlesNumber} `}</HighlightedText>
					artikler
				</Heading>
				<Heading color="white" size="huge">
					Du har brukt
					<HighlightedText>{` ${timeSpent} `}</HighlightedText> å lese
				</Heading>
				<Heading color="white" size="huge">
					Underveis du har utforsket
					<HighlightedText>{` ${tagsNumber} `}</HighlightedText>
					forskjellige temaer
				</Heading>
			</Texting>
			<NewsReaderImage />
		</ArticlesReadSection>
	)
}

Main.propTypes = {
	userData: PropTypes.shape({
		data: PropTypes.arrayOf(PropTypes.shape({ title: PropTypes.string })).isRequired,
	}).isRequired,
	statisticType: PropTypes.string.isRequired,
}

export default Main
