# Test your metadata validity on the main (master) branch

To avoid any problem when creating a release and interacting with
Zenodo, we propose a small CI snippet to check the validity of your metadata
in a `codemeta.json` file or a `.zenodo.json` file when they are modified
through a merge (pull) request to the main branch of your repository.

## GitLab

Please note that this snippet will only work if you have linked your
gitlab account with Zenodo. If not, please follow this [tutorial](
https://escape2020.pages.in2p3.fr/wp3/eossr/gitlab_to_zenodo.html#link-your-gitlab-project-with-zenodo
)

### Test your `codemeta.json` file
```yaml
stages:
 - test_codemeta

test_codemeta_ingestion:
  stage: test_codemeta
  image: gitlab-registry.in2p3.fr/escape2020/wp3/eossr:v0.6
  script:
    - eossr-check-connection-zenodo --token $ZENODO_TOKEN -p $CI_PROJECT_DIR
  rules:
    - if: '$CI_PIPELINE_SOURCE == "merge_request_event" && $CI_MERGE_REQUEST_TARGET_BRANCH_NAME == $CI_DEFAULT_BRANCH'
    - changes:
      - codemeta.json
```

### Test your `.zenodo.json` file

```yaml
stages:
 - test_zenodo

test_codemeta_ingestion:
  stage: test_codemeta
  image: gitlab-registry.in2p3.fr/escape2020/wp3/eossr:v0.6
  script:
    - eossr-check-connection-zenodo --token $ZENODO_TOKEN -p $CI_PROJECT_DIR
  rules:
    - if: '$CI_PIPELINE_SOURCE == "merge_request_event" && $CI_MERGE_REQUEST_TARGET_BRANCH_NAME == $CI_DEFAULT_BRANCH'
    - changes:
      - .zenodo.json
```

## GitHub

If you are hosting your developments on GitHub, you are probably using [the GitHub-Zenodo integration](https://docs.github.com/en/repositories/archiving-a-github-repository/referencing-and-citing-content) to upload your code to Zenodo when making a GitHub release.

The GitHub-Zenodo integration can use three sources for its metadata:
1. GitHub - if no other source of metadata exist, information such as release title and notes, versioning, contributors, etc... will be provided by GitHub metadata. Metadata from former releases or edited on the Zenodo web portal are **updated**.
2. `CITATION.cff` - if a `CITATION.cff` file lives at the root of your repository, it will be used by Zenodo to **update** existing metadata (from a former release or from GitHub)
3. `.zenodo.json` - if a `.zenodo.json` file lives at the root of your repository, it will be used by Zenodo as **unique** source of metadata and will **replace** all existing metadata 

In order to have a unique source of metadata for your project and depending on your workflow, you _might_ want to to convert your `codemeta.json` file to a `.zenodo.json` file before releases using [the `ossr-codemeta2zenodo` CLI](../docstring_sources/eossr_cli/eossr-codemeta2zenodo).


### Test your `.zenodo.json` file

First, you will need to:
- [create an access token in Zenodo](https://zenodo.org/account/settings/applications/tokens/new/)
(or [sandbox](https://sandbox.zenodo.org/account/settings/applications/tokens/new/)) with a `deposit:write` access
- add it to [your repository secrets](https://docs.github.com/en/actions/security-guides/encrypted-secrets)
under the name `ZENODO_TOKEN`.

Then, you can use the following GitHub action snippet to check the validity of your `.zenodo.json` file.
To do so, add this code snippet in a file `.github/workflows/check_zenodo.yml` in your repository:

```yaml
name: check zenodo

on:
  push:
    paths:
      - .zenodo.json

jobs:
  build:
    runs-on: ubuntu-latest
    container:
      image: gitlab-registry.in2p3.fr/escape2020/wp3/eossr:v0.6
    steps:
      - uses: actions/checkout@v2
      - name: check zenodo
        run: |
          eossr-check-connection-zenodo --token ${{ secrets.ZENODO_TOKEN }} -p .
```
