# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['asynced']

package_data = \
{'': ['*']}

install_requires = \
['typing-extensions>=4.2,<5.0']

setup_kwargs = {
    'name': 'asynced',
    'version': '1.2.0',
    'description': 'Async python for Event-Driven applications',
    'long_description': '# AsyncED\n\n-----\n\n[![PyPI version shields.io](https://img.shields.io/pypi/v/asynced.svg)](https://pypi.python.org/pypi/asynced/)\n[![PyPI pyversions](https://img.shields.io/pypi/pyversions/asynced.svg)](https://pypi.python.org/pypi/asynced/)\n[![PyPI license](https://img.shields.io/pypi/l/asynced.svg)](https://pypi.python.org/pypi/asynced/)\n\n-----\n\n**Async** python for **E**vent-**D**riven applications\n\n## Installation\n\n```bash\npip install asynced\n```\n\n## Usage\n\nAt the core of `asynced` lies `StateVar`. It can (but does not have to) contain \na value, can be watched for changed, and can easily be mapped to other \n`StateVar` instances.\n\n\n*For the sake of brevity, next examples are assumed to be run from within an async function, as `StateVar` requires a running event loop.*.\n\nCreating an empty state var is as easy as:\n\n```pycon\n>>> from asynced import StateVar\n>>> spam = StateVar()\n```\n\nWe can check whether it is set:\n\n```pycon\n>>> spam.is_set\nFalse\n```\n\nLet\'s create an `asyncio.Task` that prints the spam value once it\'s set.\n\n```pycon\n>>> async def print_spam():\n...     print(await spam)\n... \n... task = asyncio.create_task(print_spam)\n```\n\nNothing is printed yet until we set the statevar:\n\n```pycon\n>>> spam.set(\'Spam!\')\nSpam!\n```\n\nAfter we set the statevar, we see that `print_spam` printed the value.\n\nUp to here, we\'ve done nothing that `asyncio.Future` cannot do. But an \n`asyncio.Future` can only be set once, whereas a `StateVar` can be set to a \ndifferent value as much as needed. \n\nWe can watch for any changes of a statevar by async iteration:\n\n```pycon\n>>> async def print_spam_changes():\n...     async for value in spam:\n...         print(value)\n... \n... task = asyncio.create_task(print_spam_changes)\n```\n\nNow any time we set `spam` to a different value, it\'ll be printed:\n\n```pycon\n>>> spam.set(\'Spam and ham!\')\nSpam and ham!\n>>> spam.set(\'Spam and eggs!\')\nSpam and eggs!\n```\n\nNeat eh?\n\nA `StateVar` can also be constructed by passing async iterable, making it \nbehave more like an `asyncio.Task`:\n\n```pycon\n>>> async def slowrange(*args):\n...     for i in range(*args):\n...         await asyncio.sleep(1)\n...         yield i\n...\n>>> count4 = StateVar(slowrange(4))\n>>> await asyncio.sleep(3.14)\n>>> await count4\n2\n```\n\nAs we see here, the statevar will set itself to the values from the async \niterable in the background automatically. \n\n### Mapping\n\n`StateVar`\'s can also be constructed by applying a function to another \n`StateVar`:\n\n```pycon\n>>> count4_inv = StateVar(slowrange(4)).map(lambda i: -i)\n>>> async for i in count4_inv:\n...     print(i)\n```\n\nNow with one-second intervals, `0`, `-1`, `-2` and `-3` will be printed.\n\n`StateVar.map` only works for functions with a single argument, i.e. for one \nstatevar at a time. But fear not, mapping multiple statevars together is \npossible by using `asynced.statefunction`. It transforms any function \n`(a: A, b: B, ...) -> R` into one that accepts `State`\'s (`State` is the superclass of `StateVar`) and returns a \nnew `StateVar` (or some other that subclasses `State`), \n`(a: State[A], b: State[B], ...) -> StateVar[R]`. \n\n```pycon\n>>> from asynced import statefunction\n>>> @statefunction\n... def sadd(_a: float, _b: float) -> float:\n...    return _a + _b\n... \n>>> a, b = StateVar(), StateVar()\n>>> c = sadd(a, b)\n```\n\nHere, `c` will be set iff both `a` and `b` are set:\n\n```pycon\n>>>import calendar c.is_set\nFalse\n>>> a.set(12)\nc.is_set\nFalse\n>>> b.set(30)\n>>> await c\n42\n```\n\nNow, every time that `a` or `b` change, `c` will change as well.\n\n### `StateTuple`\n\nIn the same way as `StateVar`, a `StateTuple` can be awaited to get the current\nvalue once it\'s available, and async-iterated to get the values once they are \nset. Additionally, it can also be used as a tuple of individual `StateVar` \ninstances. \n\n```pycon\n>>> st = StateTuple(2)  # equivalent StateTuple(StateVar(), StateVar())\n>>> st[0].set(\'spam\')\n>>> st[1].set(\'ham\')\n>>> await st\n(\'spam\', \'ham\')\n>>> await st[-1]\n\'ham\'\n>>> st[1] = \'eggs\'  # equivalent to st[1].set(\'eggs\')\n>>> await st\n(\'spam\', \'eggs\')\n>>> s0, s1 = st\n>>> await s1\n\'eggs\'\n>>> st2 = 2 * st  # equivalent to StateTuple((st[0], st[1], st[0], st[1]))\n>>> st2[2] = \'bacon\'\n>>> await st2\n(\'bacon\', \'eggs\', \'bacon\', \'eggs\')\n>>> await st\n(\'bacon\', \'eggs\')\n```\n\n### `StateDict`\n\nLike `StateTuple`, a `StateDict` is a collection of individual `StateVar`\'s. \nIt is more similar to a `collections.defaultdict` than a regular dict, because\naccessing keys without values will return an empty `StateVar`.\n\n```pycon\n>>> sd = StateDict()\n>>> await sd\n{}\n>>> sd[\'spam\'] = \'Spam!\'\n>>> await sd\n{\'spam\': \'Spam!\'}\n>>> ham = sd[\'ham\']\n>>> ham.is_set\nFalse\n>>> list(sd.keys())\n[\'spam\']\n>>> sd[\'ham\'] = \'Ham and eggs!\'\n>>> await ham\n\'Ham and eggs!\'\n```\n\n### `.get()`\n\n`StateVar`, `StateTuple` and `StateDict` all implement a `.get()` method,\nthat can be used to get the current value, without having to use `await`.\nIf no value has been set, it will raise a `LookupError`. Alternatively,\nyou can pass a `default` value, that is to be returned in case no value is set.\n\n### Error handling\n\nIf the async iterable used to create e.g. a `StateVar` raises an error,\nthe `.is_error` property will become `True`, and the error will be raised when\nits awaited, or when`.get` or `.map` are called. The exception will propagate\nto its mapped "child" `State`\'s.\n\nIf the async iterable is exhausted (or raises `StopAsyncIteration` directly),\n`.is_stopped` will be set to `True` instead, and `StopAsyncIteration` will be **only** reraised for the waiters of `__aiter__` and `__anext__`.\n\n## API reference\n\n*~ coming soon ~*\n\n',
    'author': 'Joren Hammudoglu',
    'author_email': 'jhammudoglu@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/jorenham/asynced',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4',
}


setup(**setup_kwargs)
