# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['requests_tracker',
 'requests_tracker.headers',
 'requests_tracker.sql',
 'requests_tracker.templatetags']

package_data = \
{'': ['*'],
 'requests_tracker': ['static/requests_tracker/css/*',
                      'static/requests_tracker/sass/*',
                      'templates/*',
                      'templates/components/*',
                      'templates/partials/*']}

install_requires = \
['django>=3.2', 'sqlparse>=0.4.3']

setup_kwargs = {
    'name': 'requests-tracker',
    'version': '0.2.0',
    'description': 'The Django Requests Tracker is designed for local Django development, particularly for Rest API development. It provides various debugging information, such as SQL queries, headers, and status codes.',
    'long_description': '# Django Requests Tracker\n\n<img src="https://img.shields.io/endpoint?url=https://gist.githubusercontent.com/bensi94/0fbe0bd93d1307b7bb1096206b7817fa/raw/covbadge.json" alt="Test">\n\nA convenient Django development tool based on the great [`Django Debug Toolbar`](https://github.com/jazzband/django-debug-toolbar) but aimed towards rest API development. It collects and displays information on requests, responses, SQL queries, headers, Django settings and more.\n\n## Table of contents\n1. [Features](#features)\n    1. [Requests list](#requests-list)\n    2. [Request details](#request-details)\n2. [The example Project](#the-example-project)\n3. [Installation](#installation)\n    1. [Install the package](#install-the-package)\n    2. [Configure project settings](#configure-project-settings)\n    3. [Configure URLs](#configure-urls)\n    4. [Optional: Configure static content for WSGI and ASGI servers, e.g. Uvicorn for async Django](#configure-static-content)\n4. [Configuration](#configuration)\n   1. [IGNORE_SQL_PATTERNS](#ignore_sql_patterns)\n   2. [IGNORE_PATHS_PATTERNS](#ignore_paths_patterns)\n   3. [ENABLE_STACKTRACES](#enable_stacktraces)\n   4. [HIDE_IN_STACKTRACES](#hide_in_stacktraces)\n   5. [SQL_WARNING_THRESHOLD](#sql_warning_threshold)\n   6. [TRACK_SQL](#track_sql)\n\n## Features\n\n### Requests list\n\nDjango Requests Tracker registers every request sent to your Django application and displays them in a tidy list. Each element in the list contains information about the request\'s HTTP method, path, Django view, status code, database information and query count and execution time and duration.\n\nThe requests list can be:\n* Searched by *path*, *Django view*, *sql* and *headers*. The search is quite simple and a request is only filtered from the list if the search term does not exist in any of theses elements.\n* Ordered in ascending and descending order by *time*, *duration*, *Django view*, *query count*, *similar query count* and *duplicate query count*.\n* Auto-refreshed so that new requests will automatically show up in the list.\n* Manually refreshed.\n* Cleared.\n\n#### The requests list in action 🎥\n\n![requests-list](https://user-images.githubusercontent.com/20007971/215617783-5511c6cd-0e99-4d0d-8260-e269b7977c87.gif)\n\n### Request details\n\nEach list element can be clicked where further information collected about the request such as SQL queries and headers can be found.\n\n#### SQL queries\n\nIn request details, every SQL query executed in the context of the Django request should be shown, along with the execution time and a timeline bar that shows how big a chunk of the total time belongs to the given query. A stacktrace is shown for each query that helps with finding the origin of it.\n\nSome queries are labelled with a tag `X similar queries` or `X duplicate queries` this can often indicate a problem and can be very handy when debugging or in development.\n\n* `Similar Queries` means that the same query is executed more than once but with different parameters. This can for example happen when iterating over a list of IDs and fetching one item by ID at a time.\n* `Duplicate Queries` means that the exact same query with the same parameters is executed more than once. This can for example happen when iterating over a list child items and fetching same parent multiple times. Also known as an N-plus-1 query which is quite common problem with ORMs.\n\n#### The request details view in action 🎥\n![request-details](https://user-images.githubusercontent.com/20007971/215625549-50a0e1e1-f5f2-47c1-a36e-bb5a7cb9fd75.gif)\n\n\n### Django Settings\n\nDjango settings very often contain some logic, and usage of environment variables and can even be spread out over multiple files. So it can be very beneficial to be able to see the current computed settings being used in the running process. Django Requests Tracker offers a simple way to view this. The view can be accessed by clicking on `Django settings` in the right corner of the requests tracker view.\n\nAll information determined to be sensitive, such as keys and passwords, are masked in the displayed settings.\n\n<img width="1470" alt="Screenshot 2023-01-31 at 00 24 32" src="https://user-images.githubusercontent.com/20007971/215627287-4d62cc7d-1679-4fee-ad20-c52b59dccf34.png">\n\n## The Example Project\n\nThis repository includes an [example project](example_project) to try out the package and see how it works. It can also be a great reference when adding the package to your project. To try it out, clone this project and follow the instructions on the [example project README](example_project/README.md)\n\n## Installation\n\nIf any of the following steps are unclear, check out the [Example Project](example_project) for reference.\n\n### Install the package\n\n```bash\npip install requests-tracker\n```\n\nor install with you\'re chosen package tool, e.g.\n[poetry](https://python-poetry.org/),\n[pipenv](https://pipenv.pypa.io/en/latest/), etc.\n\n### Configure project settings\n\n#### Settings prerequisites\n\nFirst, ensure that `django.contrib.staticfiles` is in your `INSTALLED_APPS` setting and configured properly:\n```python\nINSTALLED_APPS = [\n    # ...\n    "django.contrib.staticfiles",\n    # ...\n]\n\nSTATIC_URL = "static/"\n```\n\nSecond, ensure that your `TEMPLATES` setting contains a `DjangoTemplates` backend whose `APP_DIRS` options is set to True:\n```python\nTEMPLATES = [\n    {\n        "BACKEND": "django.template.backends.django.DjangoTemplates",\n        "APP_DIRS": True,\n        # ...\n    }\n]\n```\n\n#### Install the app, add middleware and configure internal ips\n\n* Add `requests_tracker` to your `INSTALLED_APPS` setting.\n* Add `requests_tracker.middleware.requests_tracker_middleware` to your `MIDDLEWARE` setting.\n* Add your internal IP addresses to `INTERNAL_IPS` setting.\n```python\nif DEBUG:\n    INSTALLED_APPS += ["requests_tracker"]\n    MIDDLEWARE += ["requests_tracker.middleware.requests_tracker_middleware"]\n    INTERNAL_IPS = ["127.0.0.1"]\n```\n\n🚨 ️&nbsp; It\'s recommended to only configure these settings in DEBUG mode.\nEven though Django Requests Tracker will only track requests in DEBUG mode\nit\'s still a good practice to only have it installed in DEBUG mode.\n\n### Configure URLs\n\nAdd Django Requests Tracker URLs to your project\'s URLconf:\n```python\nif settings.DEBUG:\n    urlpatterns += [path("__requests_tracker__/", include("requests_tracker.urls"))]\n```\n\n🚨️&nbsp; Again it\'s recommended to only add the URLs in DEBUG mode.\n\n\n### Optional: Configure static content for WSGI and ASGI servers, e.g. Uvicorn for async Django <a name="configure-static-content"></a>\n\n#### Add static root to settings\n```python\n# 🚨 Your project might not include BASE_DIR setting but likely some variation of it 🚨\nBASE_DIR = Path(__file__).resolve().parent.parent\n\nSTATIC_ROOT = os.path.join(BASE_DIR, "static")\n```\n\n#### Add static root URLs to your project\'s URLconf:\n```python\nurlpatterns += static(settings.STATIC_URL, document_root=settings.STATIC_ROOT)\n```\n\n#### Collect static files\n```console\npython manage.py collectstatic\n```\n\n## Configuration\n\nDjango Requests Tracker provides a few very simple settings. The settings are applied by setting `REQUESTS_TRACKER_CONFIG` setting in your `settings.py` file. `REQUESTS_TRACKER_CONFIG` takes a dictonary. Example:\n\n```python\n# settings.py\n\nREQUESTS_TRACKER_CONFIG = {\n    "IGNORE_PATHS_PATTERNS": (".*/api/keep-alive.*",),\n    "ENABLE_STACKTRACES": False",\n}\n```\n\n### `IGNORE_SQL_PATTERNS`\n\nTakes a tuple of strings. Each string is a regular expression pattern.\nIf a SQL query matches any of the patterns it will be ignored and not\nshown in the requests list or request details.\n\nDefault: `()`\n\nExample:\n```python\nREQUESTS_TRACKER_CONFIG = {\n    "IGNORE_SQL_PATTERNS": (\n        r"^SELECT .* FROM django_migrations WHERE app = \'requests_tracker\'",\n        r"^SELECT .* FROM django_migrations WHERE app = \'auth\'",\n    ),\n}\n```\n\n### `IGNORE_PATHS_PATTERNS`\n\nTakes a tuple of strings. Each string is a regular expression pattern.\nIf a request path matches any of the patterns it will be ignored and not tracked.\n\nDefault: `()`\n\nExample:\n```python\nREQUESTS_TRACKER_CONFIG = {\n    "IGNORE_PATHS_PATTERNS": (\n        r".*/api/keep-alive.*",\n    ),\n}\n```\n\n### `SQL_WARNING_THRESHOLD`\n\nRepresents the threshold in milliseconds after which a SQL query is considered slow and\nwill be marked with a warning label in the SQL list.\n\nDefault: `500` (500 milliseconds)\n\nExample:\n```python\nREQUESTS_TRACKER_CONFIG = {\n    "SQL_WARNING_THRESHOLD": 50,\n}\n```\n\n### `ENABLE_STACKTRACES`\n\nIf set to `False` stacktraces will not be shown in the request details view.\n\nDefault: `True`\n\n### `HIDE_IN_STACKTRACES`\n\nTakes a tuple of strings. Each string represents a module name. If a module name is found\nin a stacktrace that part of the stacktrace will be hidden.\n\nDefault:\n```python\n(\n     "socketserver",\n     "threading",\n     "wsgiref",\n     "requests_tracker",\n     "django.db",\n     "django.core.handlers",\n     "django.core.servers",\n     "django.utils.decorators",\n     "django.utils.deprecation",\n     "django.utils.functional",\n)\n```\n\n### `TRACK_SQL`\n\nIf set to `False` SQL queries will not be tracked.\n\nDefault: `True`\n',
    'author': 'Benedikt Óskarsson',
    'author_email': 'bensi94@hotmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/bensi94/Django-Requests-Tracker',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.10,<4.0',
}


setup(**setup_kwargs)
