"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const s3 = require("@aws-cdk/aws-s3");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
test('s3 bucket with default params', () => {
    const stack = new core_1.Stack();
    new s3.Bucket(stack, 'test-s3-defaults', defaults.DefaultS3Props());
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test s3Bucket override versioningConfiguration', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        versioningConfiguration: {
            status: 'Disabled'
        },
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.CfnBucket(stack, 'test-s3-verioning', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        VersioningConfiguration: {
            Status: 'Disabled'
        }
    });
});
test('test s3Bucket override bucketEncryption', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        bucketEncryption: {
            serverSideEncryptionConfiguration: [{
                    serverSideEncryptionByDefault: {
                        kmsMasterKeyId: 'mykeyid',
                        sseAlgorithm: 'aws:kms'
                    }
                }]
        },
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.CfnBucket(stack, 'test-s3-encryption', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [{
                    ServerSideEncryptionByDefault: {
                        KMSMasterKeyID: 'mykeyid',
                        SSEAlgorithm: 'aws:kms'
                    }
                }]
        },
    });
});
test('test s3Bucket override publicAccessBlockConfiguration', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        publicAccessBlockConfiguration: {
            blockPublicAcls: false,
            blockPublicPolicy: true,
            ignorePublicAcls: false,
            restrictPublicBuckets: true
        },
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.CfnBucket(stack, 'test-s3-publicAccessBlock', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: false,
            BlockPublicPolicy: true,
            IgnorePublicAcls: false,
            RestrictPublicBuckets: true
        },
    });
});
test('test s3Bucket add lifecycleConfiguration', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        lifecycleConfiguration: {
            rules: [
                {
                    status: 'Enabled',
                    expirationInDays: 365,
                }
            ]
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.CfnBucket(stack, 'test-s3-lifecycle', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        LifecycleConfiguration: {
            Rules: [
                {
                    Status: 'Enabled',
                    ExpirationInDays: 365,
                }
            ]
        }
    });
});
test('test s3Bucket add objectLock', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        objectLockConfiguration: {
            objectLockEnabled: 'Enabled',
            rule: {
                defaultRetention: {
                    days: 365
                }
            }
        },
        objectLockEnabled: true,
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.CfnBucket(stack, 'test-s3-objlock', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        ObjectLockConfiguration: {
            ObjectLockEnabled: 'Enabled',
            Rule: {
                DefaultRetention: {
                    Days: 365
                }
            }
        },
        ObjectLockEnabled: true
    });
});
test('test s3Bucket override serverAccessLogsBucket', () => {
    const stack = new core_1.Stack();
    const myLoggingBucket = new s3.Bucket(stack, 'MyS3LoggingBucket', defaults.DefaultS3Props());
    const myS3Props = defaults.DefaultS3Props(myLoggingBucket);
    defaults.buildS3Bucket(stack, {
        bucketProps: myS3Props
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "MyS3LoggingBucket119BE896"
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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