"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
const aws_s3_1 = require("@aws-cdk/aws-s3");
test('s3 bucket with default params', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {});
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('s3 bucket with default params and bucket names', () => {
    const stack = new core_1.Stack();
    const s3BucketProps = {
        bucketName: 'my-bucket'
    };
    defaults.buildS3Bucket(stack, {
        bucketProps: s3BucketProps
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check exception for Missing existingBucketObj from props for deploy = false', () => {
    const stack = new core_1.Stack();
    try {
        defaults.buildS3Bucket(stack, {});
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('s3 bucket with bucketId', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {}, 'my');
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "myS3LoggingBucketDE461344"
            }
        },
    }));
});
test('s3 bucket with bucketProps', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            bucketName: 'mybucket'
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketName: "mybucket"
    }));
});
test('s3 bucket with access logging configured', () => {
    const stack = new core_1.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket', {
        serverAccessLogsBucket: new aws_s3_1.Bucket(stack, 'myaccesslogbucket', {})
    });
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsBucket: mybucket
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "mybucket160F8132"
            }
        },
    }));
});
test('Check S3 Bucket policy', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Sid: 'HttpsOnly',
                    Action: "*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: "*",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::GetAtt": [
                                        "S3Bucket07682993",
                                        "Arn"
                                    ]
                                },
                                "/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    }));
});
//# sourceMappingURL=data:application/json;base64,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