"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const api = require("@aws-cdk/aws-apigateway");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("../index");
const s3 = require("@aws-cdk/aws-s3");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
require("@aws-cdk/assert/jest");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
test('cloudfront distribution for ApiGateway with default params', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const func = new lambda.Function(stack, 'LambdaFunction', lambdaFunctionProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi', {
        handler: func
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test cloudfront for Api Gateway with user provided logging bucket', () => {
    const stack = new core_1.Stack();
    const loggingBucket = new s3.Bucket(stack, 'MyCloudfrontLoggingBucket', defaults.DefaultS3Props());
    const inProps = {
        code: lambda.Code.asset(`${__dirname}/lambda-test`),
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler'
    };
    const cfdProps = {
        loggingConfig: {
            bucket: loggingBucket
        }
    };
    const func = defaults.deployLambdaFunction(stack, inProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi1', {
        handler: func
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api, cfdProps);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "MyCloudfrontLoggingBucket9AA652E8",
                        "RegionalDomainName"
                    ]
                },
                IncludeCookies: false
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        HTTPPort: 80,
                        HTTPSPort: 443,
                        OriginKeepaliveTimeout: 5,
                        OriginProtocolPolicy: "https-only",
                        OriginReadTimeout: 30,
                        OriginSSLProtocols: [
                            "TLSv1.2"
                        ]
                    },
                    DomainName: {
                        "Fn::Select": [
                            0,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        "Fn::Select": [
                                            1,
                                            {
                                                "Fn::Split": [
                                                    "://",
                                                    {
                                                        "Fn::Join": [
                                                            "",
                                                            [
                                                                "https://",
                                                                {
                                                                    Ref: "RestApi1480AC499"
                                                                },
                                                                ".execute-api.",
                                                                {
                                                                    Ref: "AWS::Region"
                                                                },
                                                                ".",
                                                                {
                                                                    Ref: "AWS::URLSuffix"
                                                                },
                                                                "/",
                                                                {
                                                                    Ref: "RestApi1DeploymentStageprod4FFC9BB4"
                                                                },
                                                                "/"
                                                            ]
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: "origin1"
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true
            }
        }
    });
});
test('test cloudfront for Api Gateway override properties', () => {
    const stack = new core_1.Stack();
    const inProps = {
        code: lambda.Code.asset(`${__dirname}/lambda-test`),
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler'
    };
    const func = defaults.deployLambdaFunction(stack, inProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi1', {
        handler: func
    });
    const props = {
        originConfigs: [{
                customOriginSource: {
                    domainName: _api.url,
                    originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        allowedMethods: cloudfront.CloudFrontAllowedMethods.ALL,
                        cachedMethods: cloudfront.CloudFrontAllowedCachedMethods.GET_HEAD_OPTIONS
                    }]
            }]
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api, props);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "DELETE",
                    "GET",
                    "HEAD",
                    "OPTIONS",
                    "PATCH",
                    "POST",
                    "PUT"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "RegionalDomainName"
                    ]
                },
                IncludeCookies: false
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        HTTPPort: 80,
                        HTTPSPort: 443,
                        OriginKeepaliveTimeout: 5,
                        OriginProtocolPolicy: "http-only",
                        OriginReadTimeout: 30,
                        OriginSSLProtocols: [
                            "TLSv1.2"
                        ]
                    },
                    DomainName: {
                        "Fn::Join": [
                            "",
                            [
                                "https://",
                                {
                                    Ref: "RestApi1480AC499"
                                },
                                ".execute-api.",
                                {
                                    Ref: "AWS::Region"
                                },
                                ".",
                                {
                                    Ref: "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    Ref: "RestApi1DeploymentStageprod4FFC9BB4"
                                },
                                "/"
                            ]
                        ]
                    },
                    Id: "origin1"
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWRmcm9udC1kaXN0cmlidXRpb24tYXBpLWdhdGV3YXktaGVscGVyLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjbG91ZGZyb250LWRpc3RyaWJ1dGlvbi1hcGktZ2F0ZXdheS1oZWxwZXIudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsNENBQTZDO0FBQzdDLHdDQUFzQztBQUN0QyxzREFBc0Q7QUFDdEQsK0NBQStDO0FBQy9DLDhDQUE4QztBQUM5QyxxQ0FBcUM7QUFDckMsc0NBQXNDO0FBQ3RDLDBGQUE0RjtBQUM1RixnQ0FBOEI7QUFDOUIsNERBQStEO0FBRS9ELElBQUksQ0FBQyw0REFBNEQsRUFBRSxHQUFHLEVBQUU7SUFDdEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLG1CQUFtQixHQUF5QjtRQUNoRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO1FBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO0tBQy9DLENBQUM7SUFFRixNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFLG1CQUFtQixDQUFDLENBQUM7SUFDL0UsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7UUFDbkQsT0FBTyxFQUFFLElBQUk7S0FDZCxDQUFDLENBQUM7SUFDSCxvRUFBbUMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDakQsTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUMvRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtRUFBbUUsRUFBRSxHQUFHLEVBQUU7SUFDN0UsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLGFBQWEsR0FBYyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLDJCQUEyQixFQUFFLFFBQVEsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO0lBRTlHLE1BQU0sT0FBTyxHQUF5QjtRQUNwQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxTQUFTLGNBQWMsQ0FBQztRQUNuRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO1FBQ2xDLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLFFBQVEsR0FBRztRQUNmLGFBQWEsRUFBRTtZQUNiLE1BQU0sRUFBRSxhQUFhO1NBQ3RCO0tBQ0YsQ0FBQztJQUVGLE1BQU0sSUFBSSxHQUFHLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFFM0QsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDcEQsT0FBTyxFQUFFLElBQUk7S0FDZCxDQUFDLENBQUM7SUFFSCxvRUFBbUMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQzNELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywrQkFBK0IsRUFBRTtRQUNoRSxrQkFBa0IsRUFBRTtZQUNsQixvQkFBb0IsRUFBRTtnQkFDcEIsY0FBYyxFQUFFO29CQUNkLEtBQUs7b0JBQ0wsTUFBTTtpQkFDUDtnQkFDRCxhQUFhLEVBQUU7b0JBQ2IsS0FBSztvQkFDTCxNQUFNO2lCQUNQO2dCQUNELFFBQVEsRUFBRSxJQUFJO2dCQUNkLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsT0FBTyxFQUFFLE1BQU07cUJBQ2hCO29CQUNELFdBQVcsRUFBRSxLQUFLO2lCQUNuQjtnQkFDRCxjQUFjLEVBQUUsU0FBUztnQkFDekIsb0JBQW9CLEVBQUUsbUJBQW1CO2FBQzFDO1lBQ0QsaUJBQWlCLEVBQUUsWUFBWTtZQUMvQixPQUFPLEVBQUUsSUFBSTtZQUNiLFdBQVcsRUFBRSxPQUFPO1lBQ3BCLFdBQVcsRUFBRSxJQUFJO1lBQ2pCLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUU7b0JBQ04sWUFBWSxFQUFFO3dCQUNaLG1DQUFtQzt3QkFDbkMsb0JBQW9CO3FCQUNyQjtpQkFDRjtnQkFDRCxjQUFjLEVBQUUsS0FBSzthQUN0QjtZQUNELE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxrQkFBa0IsRUFBRTt3QkFDbEIsUUFBUSxFQUFFLEVBQUU7d0JBQ1osU0FBUyxFQUFFLEdBQUc7d0JBQ2Qsc0JBQXNCLEVBQUUsQ0FBQzt3QkFDekIsb0JBQW9CLEVBQUUsWUFBWTt3QkFDbEMsaUJBQWlCLEVBQUUsRUFBRTt3QkFDckIsa0JBQWtCLEVBQUU7NEJBQ2xCLFNBQVM7eUJBQ1Y7cUJBQ0Y7b0JBQ0QsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRTs0QkFDWixDQUFDOzRCQUNEO2dDQUNFLFdBQVcsRUFBRTtvQ0FDWCxHQUFHO29DQUNIO3dDQUNFLFlBQVksRUFBRTs0Q0FDWixDQUFDOzRDQUNEO2dEQUNFLFdBQVcsRUFBRTtvREFDWCxLQUFLO29EQUNMO3dEQUNFLFVBQVUsRUFBRTs0REFDVixFQUFFOzREQUNGO2dFQUNFLFVBQVU7Z0VBQ1Y7b0VBQ0UsR0FBRyxFQUFFLGtCQUFrQjtpRUFDeEI7Z0VBQ0QsZUFBZTtnRUFDZjtvRUFDRSxHQUFHLEVBQUUsYUFBYTtpRUFDbkI7Z0VBQ0QsR0FBRztnRUFDSDtvRUFDRSxHQUFHLEVBQUUsZ0JBQWdCO2lFQUN0QjtnRUFDRCxHQUFHO2dFQUNIO29FQUNFLEdBQUcsRUFBRSxxQ0FBcUM7aUVBQzNDO2dFQUNELEdBQUc7NkRBQ0o7eURBQ0Y7cURBQ0Y7aURBQ0Y7NkNBQ0Y7eUNBQ0Y7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsRUFBRSxFQUFFLFNBQVM7aUJBQ2Q7YUFDRjtZQUNELFVBQVUsRUFBRSxnQkFBZ0I7WUFDNUIsaUJBQWlCLEVBQUU7Z0JBQ2pCLDRCQUE0QixFQUFFLElBQUk7YUFDbkM7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFEQUFxRCxFQUFFLEdBQUcsRUFBRTtJQUMvRCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sT0FBTyxHQUF5QjtRQUNwQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxTQUFTLGNBQWMsQ0FBQztRQUNuRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO1FBQ2xDLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLElBQUksR0FBRyxRQUFRLENBQUMsb0JBQW9CLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBRTNELE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQ3BELE9BQU8sRUFBRSxJQUFJO0tBQ2QsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQThDO1FBQ3ZELGFBQWEsRUFBRSxDQUFFO2dCQUNmLGtCQUFrQixFQUFFO29CQUNoQixVQUFVLEVBQUUsSUFBSSxDQUFDLEdBQUc7b0JBQ3BCLG9CQUFvQixFQUFFLHFDQUFvQixDQUFDLFNBQVM7aUJBQ3JEO2dCQUNELFNBQVMsRUFBRSxDQUFFO3dCQUNMLGlCQUFpQixFQUFFLElBQUk7d0JBQ3ZCLGNBQWMsRUFBRSxVQUFVLENBQUMsd0JBQXdCLENBQUMsR0FBRzt3QkFDdkQsYUFBYSxFQUFFLFVBQVUsQ0FBQyw4QkFBOEIsQ0FBQyxnQkFBZ0I7cUJBQzVFLENBQUU7YUFDVixDQUFFO0tBQ0osQ0FBQztJQUVGLG9FQUFtQyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFeEQsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1FBQ2hFLGtCQUFrQixFQUFFO1lBQ2hCLG9CQUFvQixFQUFFO2dCQUNwQixjQUFjLEVBQUU7b0JBQ2QsUUFBUTtvQkFDUixLQUFLO29CQUNMLE1BQU07b0JBQ04sU0FBUztvQkFDVCxPQUFPO29CQUNQLE1BQU07b0JBQ04sS0FBSztpQkFDTjtnQkFDRCxhQUFhLEVBQUU7b0JBQ2IsS0FBSztvQkFDTCxNQUFNO29CQUNOLFNBQVM7aUJBQ1Y7Z0JBQ0QsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxPQUFPLEVBQUUsTUFBTTtxQkFDaEI7b0JBQ0QsV0FBVyxFQUFFLEtBQUs7aUJBQ25CO2dCQUNELGNBQWMsRUFBRSxTQUFTO2dCQUN6QixvQkFBb0IsRUFBRSxtQkFBbUI7YUFDMUM7WUFDRCxpQkFBaUIsRUFBRSxZQUFZO1lBQy9CLE9BQU8sRUFBRSxJQUFJO1lBQ2IsV0FBVyxFQUFFLE9BQU87WUFDcEIsV0FBVyxFQUFFLElBQUk7WUFDakIsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRTtvQkFDTixZQUFZLEVBQUU7d0JBQ1osaUNBQWlDO3dCQUNqQyxvQkFBb0I7cUJBQ3JCO2lCQUNGO2dCQUNELGNBQWMsRUFBRSxLQUFLO2FBQ3RCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQO29CQUNFLGtCQUFrQixFQUFFO3dCQUNsQixRQUFRLEVBQUUsRUFBRTt3QkFDWixTQUFTLEVBQUUsR0FBRzt3QkFDZCxzQkFBc0IsRUFBRSxDQUFDO3dCQUN6QixvQkFBb0IsRUFBRSxXQUFXO3dCQUNqQyxpQkFBaUIsRUFBRSxFQUFFO3dCQUNyQixrQkFBa0IsRUFBRTs0QkFDbEIsU0FBUzt5QkFDVjtxQkFDRjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFOzRCQUNWLEVBQUU7NEJBQ0Y7Z0NBQ0UsVUFBVTtnQ0FDVjtvQ0FDRSxHQUFHLEVBQUUsa0JBQWtCO2lDQUN4QjtnQ0FDRCxlQUFlO2dDQUNmO29DQUNFLEdBQUcsRUFBRSxhQUFhO2lDQUNuQjtnQ0FDRCxHQUFHO2dDQUNIO29DQUNFLEdBQUcsRUFBRSxnQkFBZ0I7aUNBQ3RCO2dDQUNELEdBQUc7Z0NBQ0g7b0NBQ0UsR0FBRyxFQUFFLHFDQUFxQztpQ0FDM0M7Z0NBQ0QsR0FBRzs2QkFDSjt5QkFDRjtxQkFDRjtvQkFDRCxFQUFFLEVBQUUsU0FBUztpQkFDZDthQUNGO1lBQ0QsVUFBVSxFQUFFLGdCQUFnQjtZQUM1QixpQkFBaUIsRUFBRTtnQkFDakIsNEJBQTRCLEVBQUUsSUFBSTthQUNuQztTQUNGO0tBQ0osQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAxOSBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCB7IFN5bnRoVXRpbHMgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGNsb3VkZnJvbnQgZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgYXBpIGZyb20gJ0Bhd3MtY2RrL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJy4uL2luZGV4JztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBDbG91ZEZyb250RGlzdHJpYnV0aW9uRm9yQXBpR2F0ZXdheSB9IGZyb20gJy4uL2xpYi9jbG91ZGZyb250LWRpc3RyaWJ1dGlvbi1oZWxwZXInO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgeyBPcmlnaW5Qcm90b2NvbFBvbGljeSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZyb250JztcblxudGVzdCgnY2xvdWRmcm9udCBkaXN0cmlidXRpb24gZm9yIEFwaUdhdGV3YXkgd2l0aCBkZWZhdWx0IHBhcmFtcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgY29kZTogbGFtYmRhLkNvZGUuYXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKVxuICB9O1xuXG4gIGNvbnN0IGZ1bmMgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHN0YWNrLCAnTGFtYmRhRnVuY3Rpb24nLCBsYW1iZGFGdW5jdGlvblByb3BzKTtcbiAgY29uc3QgX2FwaSA9IG5ldyBhcGkuTGFtYmRhUmVzdEFwaShzdGFjaywgJ1Jlc3RBcGknLCB7XG4gICAgaGFuZGxlcjogZnVuY1xuICB9KTtcbiAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkZvckFwaUdhdGV3YXkoc3RhY2ssIF9hcGkpO1xuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTtcblxudGVzdCgndGVzdCBjbG91ZGZyb250IGZvciBBcGkgR2F0ZXdheSB3aXRoIHVzZXIgcHJvdmlkZWQgbG9nZ2luZyBidWNrZXQnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG5cbiAgY29uc3QgbG9nZ2luZ0J1Y2tldDogczMuQnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzdGFjaywgJ015Q2xvdWRmcm9udExvZ2dpbmdCdWNrZXQnLCBkZWZhdWx0cy5EZWZhdWx0UzNQcm9wcygpKTtcblxuICBjb25zdCBpblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBjb2RlOiBsYW1iZGEuQ29kZS5hc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYS10ZXN0YCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfNixcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICBjb25zdCBjZmRQcm9wcyA9IHtcbiAgICBsb2dnaW5nQ29uZmlnOiB7XG4gICAgICBidWNrZXQ6IGxvZ2dpbmdCdWNrZXRcbiAgICB9XG4gIH07XG5cbiAgY29uc3QgZnVuYyA9IGRlZmF1bHRzLmRlcGxveUxhbWJkYUZ1bmN0aW9uKHN0YWNrLCBpblByb3BzKTtcblxuICBjb25zdCBfYXBpID0gbmV3IGFwaS5MYW1iZGFSZXN0QXBpKHN0YWNrLCAnUmVzdEFwaTEnLCB7XG4gICAgaGFuZGxlcjogZnVuY1xuICB9KTtcblxuICBDbG91ZEZyb250RGlzdHJpYnV0aW9uRm9yQXBpR2F0ZXdheShzdGFjaywgX2FwaSwgY2ZkUHJvcHMpO1xuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uXCIsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgIEFsbG93ZWRNZXRob2RzOiBbXG4gICAgICAgICAgXCJHRVRcIixcbiAgICAgICAgICBcIkhFQURcIlxuICAgICAgICBdLFxuICAgICAgICBDYWNoZWRNZXRob2RzOiBbXG4gICAgICAgICAgXCJHRVRcIixcbiAgICAgICAgICBcIkhFQURcIlxuICAgICAgICBdLFxuICAgICAgICBDb21wcmVzczogdHJ1ZSxcbiAgICAgICAgRm9yd2FyZGVkVmFsdWVzOiB7XG4gICAgICAgICAgQ29va2llczoge1xuICAgICAgICAgICAgRm9yd2FyZDogXCJub25lXCJcbiAgICAgICAgICB9LFxuICAgICAgICAgIFF1ZXJ5U3RyaW5nOiBmYWxzZVxuICAgICAgICB9LFxuICAgICAgICBUYXJnZXRPcmlnaW5JZDogXCJvcmlnaW4xXCIsXG4gICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiBcInJlZGlyZWN0LXRvLWh0dHBzXCJcbiAgICAgIH0sXG4gICAgICBEZWZhdWx0Um9vdE9iamVjdDogXCJpbmRleC5odG1sXCIsXG4gICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgSHR0cFZlcnNpb246IFwiaHR0cDJcIixcbiAgICAgIElQVjZFbmFibGVkOiB0cnVlLFxuICAgICAgTG9nZ2luZzoge1xuICAgICAgICBCdWNrZXQ6IHtcbiAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgXCJNeUNsb3VkZnJvbnRMb2dnaW5nQnVja2V0OUFBNjUyRThcIixcbiAgICAgICAgICAgIFwiUmVnaW9uYWxEb21haW5OYW1lXCJcbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIEluY2x1ZGVDb29raWVzOiBmYWxzZVxuICAgICAgfSxcbiAgICAgIE9yaWdpbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIEN1c3RvbU9yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgSFRUUFBvcnQ6IDgwLFxuICAgICAgICAgICAgSFRUUFNQb3J0OiA0NDMsXG4gICAgICAgICAgICBPcmlnaW5LZWVwYWxpdmVUaW1lb3V0OiA1LFxuICAgICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6IFwiaHR0cHMtb25seVwiLFxuICAgICAgICAgICAgT3JpZ2luUmVhZFRpbWVvdXQ6IDMwLFxuICAgICAgICAgICAgT3JpZ2luU1NMUHJvdG9jb2xzOiBbXG4gICAgICAgICAgICAgIFwiVExTdjEuMlwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBEb21haW5OYW1lOiB7XG4gICAgICAgICAgICBcIkZuOjpTZWxlY3RcIjogW1xuICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgXCJGbjo6U3BsaXRcIjogW1xuICAgICAgICAgICAgICAgICAgXCIvXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFwiRm46OlNlbGVjdFwiOiBbXG4gICAgICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBcIkZuOjpTcGxpdFwiOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgIFwiOi8vXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFwiaHR0cHM6Ly9cIixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJSZXN0QXBpMTQ4MEFDNDk5XCJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXCIuZXhlY3V0ZS1hcGkuXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcIi5cIixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFwiL1wiLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgUmVmOiBcIlJlc3RBcGkxRGVwbG95bWVudFN0YWdlcHJvZDRGRkM5QkI0XCJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXCIvXCJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9LFxuICAgICAgICAgIElkOiBcIm9yaWdpbjFcIlxuICAgICAgICB9XG4gICAgICBdLFxuICAgICAgUHJpY2VDbGFzczogXCJQcmljZUNsYXNzXzEwMFwiLFxuICAgICAgVmlld2VyQ2VydGlmaWNhdGU6IHtcbiAgICAgICAgQ2xvdWRGcm9udERlZmF1bHRDZXJ0aWZpY2F0ZTogdHJ1ZVxuICAgICAgfVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgndGVzdCBjbG91ZGZyb250IGZvciBBcGkgR2F0ZXdheSBvdmVycmlkZSBwcm9wZXJ0aWVzJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IGluUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhLXRlc3RgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM182LFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICB9O1xuXG4gIGNvbnN0IGZ1bmMgPSBkZWZhdWx0cy5kZXBsb3lMYW1iZGFGdW5jdGlvbihzdGFjaywgaW5Qcm9wcyk7XG5cbiAgY29uc3QgX2FwaSA9IG5ldyBhcGkuTGFtYmRhUmVzdEFwaShzdGFjaywgJ1Jlc3RBcGkxJywge1xuICAgIGhhbmRsZXI6IGZ1bmNcbiAgfSk7XG5cbiAgY29uc3QgcHJvcHM6IGNsb3VkZnJvbnQuQ2xvdWRGcm9udFdlYkRpc3RyaWJ1dGlvblByb3BzID0ge1xuICAgIG9yaWdpbkNvbmZpZ3M6IFsge1xuICAgICAgY3VzdG9tT3JpZ2luU291cmNlOiB7XG4gICAgICAgICAgZG9tYWluTmFtZTogX2FwaS51cmwsXG4gICAgICAgICAgb3JpZ2luUHJvdG9jb2xQb2xpY3k6IE9yaWdpblByb3RvY29sUG9saWN5LkhUVFBfT05MWVxuICAgICAgICB9LFxuICAgICAgICBiZWhhdmlvcnM6IFsge1xuICAgICAgICAgICAgICAgIGlzRGVmYXVsdEJlaGF2aW9yOiB0cnVlLFxuICAgICAgICAgICAgICAgIGFsbG93ZWRNZXRob2RzOiBjbG91ZGZyb250LkNsb3VkRnJvbnRBbGxvd2VkTWV0aG9kcy5BTEwsXG4gICAgICAgICAgICAgICAgY2FjaGVkTWV0aG9kczogY2xvdWRmcm9udC5DbG91ZEZyb250QWxsb3dlZENhY2hlZE1ldGhvZHMuR0VUX0hFQURfT1BUSU9OU1xuICAgICAgICAgICAgfSBdXG4gICAgfSBdXG4gIH07XG5cbiAgQ2xvdWRGcm9udERpc3RyaWJ1dGlvbkZvckFwaUdhdGV3YXkoc3RhY2ssIF9hcGksIHByb3BzKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6Q2xvdWRGcm9udDo6RGlzdHJpYnV0aW9uXCIsIHtcbiAgICBEaXN0cmlidXRpb25Db25maWc6IHtcbiAgICAgICAgRGVmYXVsdENhY2hlQmVoYXZpb3I6IHtcbiAgICAgICAgICBBbGxvd2VkTWV0aG9kczogW1xuICAgICAgICAgICAgXCJERUxFVEVcIixcbiAgICAgICAgICAgIFwiR0VUXCIsXG4gICAgICAgICAgICBcIkhFQURcIixcbiAgICAgICAgICAgIFwiT1BUSU9OU1wiLFxuICAgICAgICAgICAgXCJQQVRDSFwiLFxuICAgICAgICAgICAgXCJQT1NUXCIsXG4gICAgICAgICAgICBcIlBVVFwiXG4gICAgICAgICAgXSxcbiAgICAgICAgICBDYWNoZWRNZXRob2RzOiBbXG4gICAgICAgICAgICBcIkdFVFwiLFxuICAgICAgICAgICAgXCJIRUFEXCIsXG4gICAgICAgICAgICBcIk9QVElPTlNcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgQ29tcHJlc3M6IHRydWUsXG4gICAgICAgICAgRm9yd2FyZGVkVmFsdWVzOiB7XG4gICAgICAgICAgICBDb29raWVzOiB7XG4gICAgICAgICAgICAgIEZvcndhcmQ6IFwibm9uZVwiXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgUXVlcnlTdHJpbmc6IGZhbHNlXG4gICAgICAgICAgfSxcbiAgICAgICAgICBUYXJnZXRPcmlnaW5JZDogXCJvcmlnaW4xXCIsXG4gICAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6IFwicmVkaXJlY3QtdG8taHR0cHNcIlxuICAgICAgICB9LFxuICAgICAgICBEZWZhdWx0Um9vdE9iamVjdDogXCJpbmRleC5odG1sXCIsXG4gICAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICAgIEh0dHBWZXJzaW9uOiBcImh0dHAyXCIsXG4gICAgICAgIElQVjZFbmFibGVkOiB0cnVlLFxuICAgICAgICBMb2dnaW5nOiB7XG4gICAgICAgICAgQnVja2V0OiB7XG4gICAgICAgICAgICBcIkZuOjpHZXRBdHRcIjogW1xuICAgICAgICAgICAgICBcIkNsb3VkZnJvbnRMb2dnaW5nQnVja2V0M0MzRUZBQTdcIixcbiAgICAgICAgICAgICAgXCJSZWdpb25hbERvbWFpbk5hbWVcIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIH0sXG4gICAgICAgICAgSW5jbHVkZUNvb2tpZXM6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIE9yaWdpbnM6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBDdXN0b21PcmlnaW5Db25maWc6IHtcbiAgICAgICAgICAgICAgSFRUUFBvcnQ6IDgwLFxuICAgICAgICAgICAgICBIVFRQU1BvcnQ6IDQ0MyxcbiAgICAgICAgICAgICAgT3JpZ2luS2VlcGFsaXZlVGltZW91dDogNSxcbiAgICAgICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6IFwiaHR0cC1vbmx5XCIsXG4gICAgICAgICAgICAgIE9yaWdpblJlYWRUaW1lb3V0OiAzMCxcbiAgICAgICAgICAgICAgT3JpZ2luU1NMUHJvdG9jb2xzOiBbXG4gICAgICAgICAgICAgICAgXCJUTFN2MS4yXCJcbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIERvbWFpbk5hbWU6IHtcbiAgICAgICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICAgICAgXCJcIixcbiAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICBcImh0dHBzOi8vXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJSZXN0QXBpMTQ4MEFDNDk5XCJcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBcIi5leGVjdXRlLWFwaS5cIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6UmVnaW9uXCJcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBcIi5cIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIkFXUzo6VVJMU3VmZml4XCJcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBcIi9cIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIlJlc3RBcGkxRGVwbG95bWVudFN0YWdlcHJvZDRGRkM5QkI0XCJcbiAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICBcIi9cIlxuICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIElkOiBcIm9yaWdpbjFcIlxuICAgICAgICAgIH1cbiAgICAgICAgXSxcbiAgICAgICAgUHJpY2VDbGFzczogXCJQcmljZUNsYXNzXzEwMFwiLFxuICAgICAgICBWaWV3ZXJDZXJ0aWZpY2F0ZToge1xuICAgICAgICAgIENsb3VkRnJvbnREZWZhdWx0Q2VydGlmaWNhdGU6IHRydWVcbiAgICAgICAgfVxuICAgICAgfVxuICB9KTtcblxufSk7Il19