"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDeadLetterQueue = exports.buildQueue = void 0;
// Imports
const sqs = require("@aws-cdk/aws-sqs");
const defaults = require("./sqs-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const kms_helper_1 = require("./kms-helper");
function buildQueue(scope, id, props) {
    // If an existingQueueObj is not specified
    if (!props.existingQueueObj) {
        // Setup the queue
        let queueProps;
        if (props.queueProps) {
            // If property overrides have been provided, incorporate them and deploy
            queueProps = utils_1.overrideProps(defaults.DefaultQueueProps(), props.queueProps);
        }
        else {
            // If no property overrides, deploy using the default configuration
            queueProps = defaults.DefaultQueueProps();
        }
        // Determine whether a DLQ property should be added
        if (props.deadLetterQueue) {
            queueProps.deadLetterQueue = props.deadLetterQueue;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey) {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                queueProps.encryptionMasterKey = props.encryptionKey;
            }
            else {
                queueProps.encryptionMasterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        const queue = new sqs.Queue(scope, id, queueProps);
        applySecureQueuePolicy(queue);
        // Return the queue
        return [queue, queue.encryptionMasterKey];
    }
    else {
        // If an existingQueueObj is specified, return that object as the queue to be used
        return [props.existingQueueObj];
    }
}
exports.buildQueue = buildQueue;
function buildDeadLetterQueue(props) {
    const mrc = (props.maxReceiveCount) ? props.maxReceiveCount : defaults.defaultMaxReceiveCount;
    // Setup the queue interface
    const dlq = {
        maxReceiveCount: mrc,
        queue: props.deadLetterQueue
    };
    // Return the dead letter queue
    return dlq;
}
exports.buildDeadLetterQueue = buildDeadLetterQueue;
function applySecureQueuePolicy(queue) {
    // Apply queue policy to enforce only the queue owner can perform operations on queue
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'QueueOwnerOnlyAccess',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "sqs:DeleteMessage",
            "sqs:ReceiveMessage",
            "sqs:SendMessage",
            "sqs:GetQueueAttributes",
            "sqs:RemovePermission",
            "sqs:AddPermission",
            "sqs:SetQueueAttributes"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(queue).account)],
        effect: aws_iam_1.Effect.ALLOW
    }));
    // Apply Topic policy to enforce encryption of data in transit
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "SQS:*"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
//# sourceMappingURL=data:application/json;base64,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