"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildTopic = void 0;
// Imports
const sns = require("@aws-cdk/aws-sns");
const kms = require("@aws-cdk/aws-kms");
const sns_defaults_1 = require("./sns-defaults");
const kms_helper_1 = require("./kms-helper");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
function applySecureTopicPolicy(topic) {
    // Apply topic policy to enforce only the topic owner can publish and subscribe to this topic
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'TopicOwnerOnlyAccess',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(topic).account)],
        effect: aws_iam_1.Effect.ALLOW,
        conditions: {
            StringEquals: {
                "AWS:SourceOwner": core_1.Stack.of(topic).account
            }
        }
    }));
    // Apply Topic policy to enforce encryption of data in transit
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
function buildTopic(scope, props) {
    if (!props.existingTopicObj) {
        // Setup the topic properties
        let snsTopicProps;
        if (props.topicProps) {
            // If property overrides have been provided, incorporate them and deploy
            snsTopicProps = utils_1.overrideProps(sns_defaults_1.DefaultSnsTopicProps, props.topicProps);
        }
        else {
            // If no property overrides, deploy using the default configuration
            snsTopicProps = sns_defaults_1.DefaultSnsTopicProps;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey === undefined || props.enableEncryptionWithCustomerManagedKey === false) {
            // Retrieve SNS managed key to encrypt the SNS Topic
            const awsManagedKey = kms.Alias.fromAliasName(scope, 'aws-managed-key', 'alias/aws/sns');
            snsTopicProps.masterKey = awsManagedKey;
        }
        else {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                snsTopicProps.masterKey = props.encryptionKey;
            }
            else {
                // Create a new Customer Managed KMS key
                snsTopicProps.masterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        // Create the SNS Topic
        const topic = new sns.Topic(scope, 'SnsTopic', snsTopicProps);
        applySecureTopicPolicy(topic);
        return [topic, snsTopicProps.masterKey];
    }
    else {
        return [props.existingTopicObj];
    }
}
exports.buildTopic = buildTopic;
//# sourceMappingURL=data:application/json;base64,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