"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.applySecureBucketPolicy = exports.buildS3Bucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
function buildS3Bucket(scope, props, bucketId) {
    if (props.bucketProps) {
        return s3BucketWithLogging(scope, props.bucketProps, bucketId);
    }
    else {
        return s3BucketWithLogging(scope, s3_bucket_defaults_1.DefaultS3Props(), bucketId);
    }
}
exports.buildS3Bucket = buildS3Bucket;
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function s3BucketWithLogging(scope, s3BucketProps, bucketId) {
    // Create the Application Bucket
    let bucketprops;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        bucketprops = s3_bucket_defaults_1.DefaultS3Props();
    }
    else {
        // Create the Logging Bucket
        loggingBucket = new s3.Bucket(scope, _loggingBucketId, s3_bucket_defaults_1.DefaultS3Props());
        applySecureBucketPolicy(loggingBucket);
        // Extract the CfnBucket from the loggingBucket
        const loggingBucketResource = loggingBucket.node.findChild('Resource');
        // Override accessControl configuration and add metadata for the logging bucket
        loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
        loggingBucketResource.cfnOptions.metadata = {
            cfn_nag: {
                rules_to_suppress: [{
                        id: 'W35',
                        reason: `This S3 bucket is used as the access logging bucket for another bucket`
                    }]
            }
        };
        bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
//# sourceMappingURL=data:application/json;base64,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