from enum import Enum
from dataclasses import dataclass, InitVar, field


class DipoleOrientation(Enum):
	FREE = 1
	FIXED_XY = 2
	FIXED_Z = 3


class StrainFieldType(Enum):
	ZERO_MEAN = 1
	NON_ZERO_MEAN = 2


@dataclass
class ConfigParams:
	"""
	Represents some config params

	Parameters
	----------
	count : int
	The number of dipoles generated by this set of parameters.

	dipole_magnitude : float
	The log 10 magnitude of the dipoles.

	w : float
	The oscillation frequency.
	"""

	count: int
	dipole_magnitude: float
	orientation: DipoleOrientation = field(init=False)
	orientation_raw: InitVar[int]
	strain_field_type: StrainFieldType = field(init=False)
	strain_field_type_raw: InitVar[int]
	strain_field_characteristic_size: float

	def __post_init__(self, orientation_raw, strain_field_type_raw):
		self.orientation = DipoleOrientation(orientation_raw)
		self.strain_field_type = StrainFieldType(strain_field_type_raw)


def dict_to_params(json_dict):
	"""
	Converts a dict obtained as for example from json parsing to a ConfigParams object.

	Expects fields named "count", "dipole_magnitude", "orientation", "strain_field_type" and "strain_field_characteristic_size",
	with values doing the obvious thing wrt to the types above (and with the same sort of enum numbering.)
	"""
	return ConfigParams(
		json_dict["count"],
		json_dict["dipole_magnitude"],
		json_dict["orientation"],
		json_dict["strain_field_type"],
		json_dict["strain_field_characteristic_size"],
	)
