"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CreateTestCache = exports.suppressAutoDeleteHandlerWarnings = exports.getFakeCertificate = exports.getTestVpc = exports.generateIntegStackName = exports.CreateScrapBucket = exports.fakeEcrRepoArn = void 0;
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const vpc_helper_1 = require("../lib/vpc-helper");
const vpc_defaults_1 = require("../lib/vpc-defaults");
const utils_1 = require("../lib/utils");
const elasticache_helper_1 = require("../lib/elasticache-helper");
const path = require("path");
const cache = require("aws-cdk-lib/aws-elasticache");
const ec2 = require("aws-cdk-lib/aws-ec2");
const acm = require("aws-cdk-lib/aws-certificatemanager");
const elasticache_defaults_1 = require("../lib/elasticache-defaults");
exports.fakeEcrRepoArn = 'arn:aws:ecr:us-east-1:123456789012:repository/fake-repo';
// Creates a bucket used for testing - minimal properties, destroyed after test
function CreateScrapBucket(scope, props) {
    const defaultProps = {
        versioned: true,
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
    };
    let synthesizedProps;
    if (props) {
        synthesizedProps = utils_1.overrideProps(defaultProps, props);
    }
    else {
        synthesizedProps = defaultProps;
    }
    const scriptBucket = new aws_s3_1.Bucket(scope, "existingScriptLocation", synthesizedProps);
    utils_1.addCfnSuppressRules(scriptBucket, [
        {
            id: "W51",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W35",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct implementation",
        },
        {
            id: "W41",
            reason: "This S3 bucket is created for unit/ integration testing purposes only and not part of       the actual construct",
        }
    ]);
    return scriptBucket;
}
exports.CreateScrapBucket = CreateScrapBucket;
/**
 * @summary Creates a stack name for Integration tests
 * @param {string} filename - the filename of the integ test
 * @returns {string} - a string with current filename after removing anything before the prefix '.' and suffix '.js'
 * e.g. 'integ.apigateway-dynamodb-CRUD.js' will return 'apigateway-dynamodb-CRUD'
 */
function generateIntegStackName(filename) {
    const file = path.basename(filename, path.extname(filename));
    const stackname = file.slice(file.lastIndexOf('.') + 1).replace(/_/g, '-');
    return stackname;
}
exports.generateIntegStackName = generateIntegStackName;
// Helper Functions
function getTestVpc(stack, publicFacing = true, userVpcProps) {
    return vpc_helper_1.buildVpc(stack, {
        defaultVpcProps: publicFacing ?
            vpc_defaults_1.DefaultPublicPrivateVpcProps() :
            vpc_defaults_1.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
            cidr: '172.168.0.0/16',
        },
        userVpcProps
    });
}
exports.getTestVpc = getTestVpc;
function getFakeCertificate(scope, id) {
    return acm.Certificate.fromCertificateArn(scope, id, "arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012");
}
exports.getFakeCertificate = getFakeCertificate;
function suppressAutoDeleteHandlerWarnings(stack) {
    aws_cdk_lib_1.Stack.of(stack).node.children.forEach(child => {
        if (child.node.id === 'Custom::S3AutoDeleteObjectsCustomResourceProvider') {
            const handlerFunction = child.node.findChild('Handler');
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W58", reason: "CDK generated custom resource" }]);
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W89", reason: "CDK generated custom resource" }]);
            utils_1.addCfnSuppressRules(handlerFunction, [{ id: "W92", reason: "CDK generated custom resource" }]);
        }
    });
}
exports.suppressAutoDeleteHandlerWarnings = suppressAutoDeleteHandlerWarnings;
function CreateTestCache(scope, id, vpc, port) {
    const cachePort = port ?? elasticache_defaults_1.GetDefaultCachePort();
    // Create the subnet group from all the isolated subnets in the VPC
    const subnetGroup = elasticache_helper_1.createCacheSubnetGroup(scope, vpc, id);
    const emptySG = new ec2.SecurityGroup(scope, `${id}-cachesg`, {
        vpc,
        allowAllOutbound: true,
    });
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W40", reason: "Test Resource" }]);
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W5", reason: "Test Resource" }]);
    utils_1.addCfnSuppressRules(emptySG, [{ id: "W36", reason: "Test Resource" }]);
    const cacheProps = {
        clusterName: `${id}-cdk-cluster`,
        cacheNodeType: "cache.t3.medium",
        engine: "memcached",
        numCacheNodes: 2,
        port: cachePort,
        azMode: "cross-az",
        vpcSecurityGroupIds: [emptySG.securityGroupId],
        cacheSubnetGroupName: subnetGroup.cacheSubnetGroupName,
    };
    const newCache = new cache.CfnCacheCluster(scope, `${id}-cluster`, cacheProps);
    newCache.addDependsOn(subnetGroup);
    return newCache;
}
exports.CreateTestCache = CreateTestCache;
//# sourceMappingURL=data:application/json;base64,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