"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("..");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const __1 = require("..");
const ec2 = require("aws-cdk-lib/aws-ec2");
const ecs = require("aws-cdk-lib/aws-ecs");
const ecr = require("aws-cdk-lib/aws-ecr");
require("@aws-cdk/assert/jest");
test('Test with all defaults', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = defaults.getTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn);
    expect(stack).toHaveResource("AWS::ECS::Service", {
        Cluster: {
            Ref: "testclusterDF8B0D19"
        },
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        DesiredCount: 2,
        EnableECSManagedTags: false,
        LaunchType: "FARGATE",
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                SecurityGroups: [
                    {
                        "Fn::GetAtt": [
                            "testsg872EB48A",
                            "GroupId"
                        ]
                    }
                ],
                Subnets: [
                    {
                        Ref: "VpcPrivateSubnet1Subnet536B997A"
                    },
                    {
                        Ref: "VpcPrivateSubnet2Subnet3788AAA1"
                    }
                ]
            }
        },
        PlatformVersion: "LATEST",
        TaskDefinition: {
            Ref: "testtaskdefF924AD58"
        }
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
            }
        ],
    });
    expect(stack).toHaveResourceLike("AWS::EC2::SecurityGroup", {
        GroupDescription: 'Construct created security group'
    });
    expect(stack).toCountResources("AWS::EC2::VPCEndpoint", 3);
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
});
test('Test with all defaults in isolated VPC', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn);
    expect(stack).toHaveResource("AWS::ECS::Service", {
        Cluster: {
            Ref: "testclusterDF8B0D19"
        },
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        DesiredCount: 2,
        EnableECSManagedTags: false,
        LaunchType: "FARGATE",
        NetworkConfiguration: {
            AwsvpcConfiguration: {
                AssignPublicIp: "DISABLED",
                SecurityGroups: [
                    {
                        "Fn::GetAtt": [
                            "testsg872EB48A",
                            "GroupId"
                        ]
                    }
                ],
                Subnets: [
                    {
                        Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                    },
                    {
                        Ref: "VpcisolatedSubnet2Subnet39217055"
                    }
                ]
            }
        },
        PlatformVersion: "LATEST",
        TaskDefinition: {
            Ref: "testtaskdefF924AD58"
        }
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
            }
        ],
    });
    expect(stack).toCountResources("AWS::EC2::VPCEndpoint", 3);
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Interface",
    });
    expect(stack).toHaveResource("AWS::EC2::VPCEndpoint", {
        VpcEndpointType: "Gateway",
    });
});
test('Test with custom task definition', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, undefined, undefined, undefined, {
        image: CreateImage(stack)
    });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        ContainerDefinitions: [
            {
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/existingImage:latest"
                        ]
                    ]
                },
            }
        ],
    });
});
test('Test with custom container definition', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn, undefined, { cpu: 256, memoryLimitMiB: 512 });
    expect(stack).toHaveResourceLike("AWS::ECS::TaskDefinition", {
        Cpu: '256',
        Memory: '512'
    });
});
test('Test with custom cluster props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const clusterName = 'test-value';
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, { clusterName }, defaults.fakeEcrRepoArn, undefined);
    expect(stack).toHaveResourceLike("AWS::ECS::Cluster", {
        ClusterName: clusterName,
    });
});
test('Test with custom Fargate Service props', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const serviceName = 'test-value';
    const testVpc = CreateIsolatedTestVpc(stack);
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { serviceName });
    expect(stack).toHaveResourceLike("AWS::ECS::Service", {
        ServiceName: serviceName,
    });
});
test('Test with custom security group', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const groupDescription = 'Test generated security group';
    const testVpc = CreateIsolatedTestVpc(stack);
    const customSg = new ec2.SecurityGroup(stack, 'custom-sg', {
        disableInlineRules: true,
        allowAllOutbound: false,
        vpc: testVpc,
        description: groupDescription
    });
    __1.CreateFargateService(stack, 'test', testVpc, undefined, defaults.fakeEcrRepoArn, undefined, undefined, undefined, { securityGroups: [customSg] });
    expect(stack).toHaveResource("AWS::EC2::SecurityGroup", {
        GroupDescription: groupDescription,
    });
    expect(stack).not.toHaveResource("AWS::EC2::SecurityGroup", {
        GroupDescription: 'Construct created security group',
    });
});
test('Test no image repo or image is an error', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const testVpc = CreateIsolatedTestVpc(stack);
    const app = () => {
        __1.CreateFargateService(stack, 'test', testVpc);
    };
    expect(app).toThrowError("Not Implemented - image without repo name and version");
});
// CheckFargateProps tests
test('Check providing existing service AND props is an error', () => {
    const props = {
        existingFargateServiceObject: { place: "holder" },
        existingImageObject: { place: "holder2" }
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("If you provide an existingFargateServiceObject, you cannot provide any props defining a new service\n");
});
test('Check providing existing image AND props is an error', () => {
    const props = {
        existingImageObject: { place: "holder" },
        ecrRepositoryArn: { place: "holder2" }
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("If you provide an existingImageObject then you cannot provide an ecrRepositoryArn nor ecrImageVersion\n");
});
test('Check providing vpc in the targetGroupsProps is an error', () => {
    const props = {
        targetGroupProps: { vpc: { place: "holder" } },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("Provide all VPC info at Construct level, not within targetGroupProps\n");
});
test('Check providing taskDefinition in the fargateServiceProps is an error', () => {
    const props = {
        fargateServiceProps: { taskDefinition: { place: "holder" } },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("The construct cannot accept an existing task definition in fargateServiceProps\n");
});
test('Check providing cluster in fargateServiceProps AND clusterProps is an error', () => {
    const props = {
        fargateServiceProps: { cluster: { place: "holder" } },
        clusterProps: { place: "holder2" },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("If you provide a cluster in fargateServiceProps then you cannot provide clusterProps\n");
});
test('Check providing vpc in clusterProps is an error', () => {
    const props = {
        clusterProps: { vpc: { place: "holder" } },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("All services in the construct use the construct VPC, you cannot specify a VPC in clusterProps\n");
});
test('Check providing existing service without existing container and existing VPC is an error', () => {
    const props = {
        existingFargateServiceObject: { place: "holder" },
        existingVpc: { place: "holder2" },
    };
    const app = () => {
        defaults.CheckFargateProps(props);
    };
    expect(app).toThrowError("If an existing Service is indicated by supplying either existingFargateServiceObject or existingContainerDefinitionObject, then existingFargateServiceObject, existingContainerDefinitionObject, and existingVpc must all be provided\n");
});
// Helper functions
function CreateIsolatedTestVpc(stack) {
    return defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
    });
}
function CreateImage(stack) {
    return ecs.ContainerImage.fromEcrRepository(ecr.Repository.fromRepositoryArn(stack, `test-container`, 
    // This is different than fakeEcrRepoArn because we're testing custom image
    "arn:aws:ecr:us-east-1:123456789012:repository/existingImage"), "latest");
}
//# sourceMappingURL=data:application/json;base64,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