"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddAwsServiceEndpoint = exports.ServiceEndpointTypes = exports.buildVpc = void 0;
const ec2 = require("aws-cdk-lib/aws-ec2");
const security_group_helper_1 = require("./security-group-helper");
const utils_1 = require("./utils");
function buildVpc(scope, props) {
    if (props?.existingVpc) {
        return props?.existingVpc;
    }
    let cumulativeProps = props?.defaultVpcProps;
    cumulativeProps = utils_1.consolidateProps(cumulativeProps, props?.userVpcProps, props?.constructVpcProps);
    const vpc = new ec2.Vpc(scope, "Vpc", cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    const flowLog = vpc.addFlowLog("FlowLog");
    SuppressMapPublicIpWarnings(vpc);
    SuppressEncryptedLogWarnings(flowLog);
    return vpc;
}
exports.buildVpc = buildVpc;
var ServiceEndpointTypes;
(function (ServiceEndpointTypes) {
    ServiceEndpointTypes["DYNAMODB"] = "DDB";
    ServiceEndpointTypes["SNS"] = "SNS";
    ServiceEndpointTypes["SQS"] = "SQS";
    ServiceEndpointTypes["S3"] = "S3";
    ServiceEndpointTypes["STEP_FUNCTIONS"] = "STEP_FUNCTIONS";
    ServiceEndpointTypes["SAGEMAKER_RUNTIME"] = "SAGEMAKER_RUNTIME";
    ServiceEndpointTypes["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ServiceEndpointTypes["SSM"] = "SSM";
    ServiceEndpointTypes["ECR_API"] = "ECR_API";
    ServiceEndpointTypes["ECR_DKR"] = "ECR_DKR";
    ServiceEndpointTypes["EVENTS"] = "CLOUDWATCH_EVENTS";
})(ServiceEndpointTypes = exports.ServiceEndpointTypes || (exports.ServiceEndpointTypes = {}));
var EndpointTypes;
(function (EndpointTypes) {
    EndpointTypes["GATEWAY"] = "Gateway";
    EndpointTypes["INTERFACE"] = "Interface";
})(EndpointTypes || (EndpointTypes = {}));
const endpointSettings = [
    {
        endpointName: ServiceEndpointTypes.DYNAMODB,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: ServiceEndpointTypes.S3,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: ServiceEndpointTypes.STEP_FUNCTIONS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.STEP_FUNCTIONS,
    },
    {
        endpointName: ServiceEndpointTypes.SNS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: ServiceEndpointTypes.SQS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SQS,
    },
    {
        endpointName: ServiceEndpointTypes.SAGEMAKER_RUNTIME,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME,
    },
    {
        endpointName: ServiceEndpointTypes.SECRETS_MANAGER,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
    },
    {
        endpointName: ServiceEndpointTypes.SSM,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SSM,
    },
    {
        endpointName: ServiceEndpointTypes.ECR_API,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.ECR
    },
    {
        endpointName: ServiceEndpointTypes.ECR_DKR,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
    },
    {
        endpointName: ServiceEndpointTypes.EVENTS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS
    }
];
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (CheckIfEndpointAlreadyExists(vpc, interfaceTag)) {
        return;
    }
    const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
    if (!service) {
        throw new Error("Unsupported Service sent to AddServiceEndpoint");
    }
    if (service.endpointType === EndpointTypes.GATEWAY) {
        AddGatewayEndpoint(vpc, service, interfaceTag);
    }
    if (service.endpointType === EndpointTypes.INTERFACE) {
        AddInterfaceEndpoint(scope, vpc, service, interfaceTag);
    }
    return;
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
function CheckIfEndpointAlreadyExists(vpc, interfaceTag) {
    return vpc.node.children.some((child) => child.node.id === interfaceTag);
}
function SuppressMapPublicIpWarnings(vpc) {
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        utils_1.addCfnSuppressRules(cfnSubnet, [
            {
                id: 'W33',
                reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true'
            }
        ]);
    });
}
function SuppressEncryptedLogWarnings(flowLog) {
    // Add Cfn Nag suppression for CloudWatchLogs LogGroups data is encrypted
    const cfnLogGroup = flowLog.logGroup?.node.defaultChild;
    utils_1.addCfnSuppressRules(cfnLogGroup, [
        {
            id: 'W84',
            reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
        }
    ]);
}
function AddInterfaceEndpoint(scope, vpc, service, interfaceTag) {
    const endpointDefaultSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, `${scope.node.id}-${service.endpointName}`, {
        vpc,
        allowAllOutbound: true,
    }, [{ peer: ec2.Peer.ipv4(vpc.vpcCidrBlock), connection: ec2.Port.tcp(443) }], []);
    vpc.addInterfaceEndpoint(interfaceTag, {
        service: service.endpointInterfaceService,
        securityGroups: [endpointDefaultSecurityGroup],
    });
}
function AddGatewayEndpoint(vpc, service, interfaceTag) {
    vpc.addGatewayEndpoint(interfaceTag, {
        service: service.endpointGatewayService,
    });
}
//# sourceMappingURL=data:application/json;base64,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