"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildDeadLetterQueue = exports.buildQueue = void 0;
// Imports
const sqs = require("aws-cdk-lib/aws-sqs");
const defaults = require("./sqs-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const kms_helper_1 = require("./kms-helper");
function buildQueue(scope, id, props) {
    // If an existingQueueObj is not specified
    if (!props.existingQueueObj) {
        // Setup the queue
        let queueProps;
        if (props.queueProps) {
            // If property overrides have been provided, incorporate them and deploy
            const checkFifo = props.queueProps.fifo ? true : undefined;
            queueProps = utils_1.overrideProps(defaults.DefaultQueueProps(), { ...props.queueProps, fifo: checkFifo });
        }
        else {
            // If no property overrides, deploy using the default configuration
            queueProps = defaults.DefaultQueueProps();
        }
        // Determine whether a DLQ property should be added
        if (props.deadLetterQueue) {
            queueProps.deadLetterQueue = props.deadLetterQueue;
        }
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey) {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                queueProps.encryptionMasterKey = props.encryptionKey;
            }
            else {
                queueProps.encryptionMasterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        const queue = new sqs.Queue(scope, id, queueProps);
        applySecureQueuePolicy(queue);
        // Return the queue
        return [queue, queue.encryptionMasterKey];
    }
    else {
        // If an existingQueueObj is specified, return that object as the queue to be used
        return [props.existingQueueObj];
    }
}
exports.buildQueue = buildQueue;
function buildDeadLetterQueue(scope, props) {
    if (!props.existingQueueObj && (props.deployDeadLetterQueue || props.deployDeadLetterQueue === undefined)) {
        // Create the Dead Letter Queue
        const [dlq] = buildQueue(scope, 'deadLetterQueue', {
            queueProps: props.deadLetterQueueProps
        });
        const mrc = (props.maxReceiveCount) ? props.maxReceiveCount : defaults.defaultMaxReceiveCount;
        // Setup the Dead Letter Queue interface
        const dlqInterface = {
            maxReceiveCount: mrc,
            queue: dlq
        };
        // Return the dead letter queue interface
        return dlqInterface;
    }
    return;
}
exports.buildDeadLetterQueue = buildDeadLetterQueue;
function applySecureQueuePolicy(queue) {
    // Apply queue policy to enforce only the queue owner can perform operations on queue
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'QueueOwnerOnlyAccess',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "sqs:DeleteMessage",
            "sqs:ReceiveMessage",
            "sqs:SendMessage",
            "sqs:GetQueueAttributes",
            "sqs:RemovePermission",
            "sqs:AddPermission",
            "sqs:SetQueueAttributes"
        ],
        principals: [new aws_iam_1.AccountPrincipal(aws_cdk_lib_1.Stack.of(queue).account)],
        effect: aws_iam_1.Effect.ALLOW
    }));
    // Apply Topic policy to enforce encryption of data in transit
    queue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${queue.queueArn}`
        ],
        actions: [
            "SQS:*"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
//# sourceMappingURL=data:application/json;base64,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