"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addCfnNagS3BucketNotificationRulesToSuppress = exports.buildS3Bucket = exports.createAlbLoggingBucket = exports.createLoggingBucket = void 0;
const s3 = require("aws-cdk-lib/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
function createLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: _reason
        }
    ]);
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function createAlbLoggingBucket(scope, bucketId, loggingBucketProps) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, loggingBucketProps);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    utils_1.addCfnSuppressRules(loggingBucketResource, [
        {
            id: 'W35',
            reason: "This is a log bucket for an Application Load Balancer"
        }
    ]);
    return loggingBucket;
}
exports.createAlbLoggingBucket = createAlbLoggingBucket;
function buildS3Bucket(scope, props, bucketId) {
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: aws_s3_1.StorageClass.GLACIER,
                    transitionAfter: aws_cdk_lib_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let customBucketProps;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    // If logging S3 access logs is enabled/undefined and an existing bucket object is not provided
    if (props.logS3AccessLogs !== false && !(props.bucketProps?.serverAccessLogsBucket)) {
        // Create the Logging Bucket
        let loggingBucketProps = s3_bucket_defaults_1.DefaultS3Props();
        if (props.loggingBucketProps) {
            // User provided logging bucket props
            loggingBucketProps = utils_1.overrideProps(loggingBucketProps, props.loggingBucketProps);
        }
        else if (props.bucketProps?.removalPolicy) {
            // If the client explicitly specified a removal policy for the main bucket,
            // then replicate that policy on the logging bucket
            loggingBucketProps = utils_1.overrideProps(loggingBucketProps, { removalPolicy: props.bucketProps.removalPolicy });
        }
        loggingBucket = createLoggingBucket(scope, _loggingBucketId, loggingBucketProps);
    }
    // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
    if (props.bucketProps?.versioned === undefined || props.bucketProps.versioned) {
        customBucketProps = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
    }
    else {
        customBucketProps = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
    }
    customBucketProps = props.bucketProps ? utils_1.overrideProps(customBucketProps, props.bucketProps) : customBucketProps;
    const s3Bucket = new s3.Bucket(scope, _bucketId, customBucketProps);
    return [s3Bucket, loggingBucket];
}
exports.buildS3Bucket = buildS3Bucket;
function addCfnNagS3BucketNotificationRulesToSuppress(stackRoot, logicalId) {
    const notificationsResourceHandler = stackRoot.node.tryFindChild(logicalId);
    const notificationsResourceHandlerRoleRole = notificationsResourceHandler.node.findChild('Role');
    const notificationsResourceHandlerRolePolicy = notificationsResourceHandlerRoleRole.node.findChild('DefaultPolicy');
    // Extract the CfnFunction from the Function
    const fnResource = notificationsResourceHandler.node.findChild('Resource');
    utils_1.addCfnSuppressRules(fnResource, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    // Extract the CfnPolicy from the iam.Policy
    const policyResource = notificationsResourceHandlerRolePolicy.node.findChild('Resource');
    utils_1.addCfnSuppressRules(policyResource, [
        {
            id: 'W12',
            reason: `Bucket resource is '*' due to circular dependency with bucket and role creation at the same time`
        }
    ]);
}
exports.addCfnNagS3BucketNotificationRulesToSuppress = addCfnNagS3BucketNotificationRulesToSuppress;
//# sourceMappingURL=data:application/json;base64,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