"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLambdaVpcSecurityGroupIds = exports.addPermission = exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("aws-cdk-lib/aws-lambda");
const iam = require("aws-cdk-lib/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("aws-cdk-lib");
const utils_1 = require("./utils");
const security_group_helper_1 = require("./security-group-helper");
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps, undefined, props.vpc);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        if (props.vpc) {
            const levelOneFunction = props.existingLambdaObj.node.defaultChild;
            if (props.lambdaFunctionProps?.securityGroups) {
                let ctr = 20;
                props.lambdaFunctionProps?.securityGroups.forEach(sg => {
                    // TODO: Discuss with someone why I can't get R/O access to VpcConfigSecurityGroupIds
                    levelOneFunction.addOverride(`Properties.VpcConfig.SecurityGroupIds.${ctr++}`, sg.securityGroupId);
                });
            }
            if (!props.existingLambdaObj.isBoundToVpc) {
                throw Error('A Lambda function must be bound to a VPC upon creation, it cannot be added to a VPC in a subsequent construct');
            }
        }
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId, vpc) {
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    if (vpc && lambdaFunctionProps.vpc) {
        throw new Error("Cannot provide a VPC in both the lambdaFunctionProps and the function argument");
    }
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // If this Lambda function is going to access resoures in a
    // VPC, then it needs privileges to access an ENI in that VPC
    if (lambdaFunctionProps.vpc || vpc) {
        lambdaServiceRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                "ec2:CreateNetworkInterface",
                "ec2:DescribeNetworkInterfaces",
                "ec2:DeleteNetworkInterface",
                "ec2:AssignPrivateIpAddresses",
                "ec2:UnassignPrivateIpAddresses"
            ],
            resources: ["*"]
        }));
    }
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    let finalLambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    if (vpc) {
        // This is literally setting up what would be the default SG, but
        // we need to to it explicitly to disable the cfn_nag error
        const lambdaSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, "ReplaceDefaultSecurityGroup", {
            vpc,
            allowAllOutbound: true,
        }, [], []);
        finalLambdaFunctionProps = utils_1.overrideProps(finalLambdaFunctionProps, {
            securityGroups: [lambdaSecurityGroup],
            vpc,
        }, true);
    }
    const lambdafunction = new lambda.Function(scope, _functionId, finalLambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_14_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    utils_1.addCfnSuppressRules(lambdafunction, [
        {
            id: 'W58',
            reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with tighter permissions.`
        },
        {
            id: 'W89',
            reason: `This is not a rule for the general case, just for specific use cases/industries`
        },
        {
            id: 'W92',
            reason: `Impossible for us to define the correct concurrency for clients`
        }
    ]);
    if (cfnLambdafunction.tracingConfig) {
        // Find the X-Ray IAM Policy
        const cfnLambdafunctionDefPolicy = lambdafunction.role?.node.tryFindChild('DefaultPolicy')?.node.findChild('Resource');
        // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
        utils_1.addCfnSuppressRules(cfnLambdafunctionDefPolicy, [
            {
                id: 'W12',
                reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray and access ENIs in a VPC.`
            }
        ]);
    }
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
// A wrapper above Function.addPermision that
// prevents two different calls to addPermission using
// the same construct id.
function addPermission(targetFunction, name, permission) {
    targetFunction.addPermission(GetNextId(targetFunction.permissionsNode.children, name), permission);
}
exports.addPermission = addPermission;
// Scan the current permissions for any entries with this core name and
// return the first available synthesized name. Names are coreName-suffix.
function GetNextId(children, coreName) {
    let lastSuffix = 0;
    children.forEach(child => {
        // if (compare right side of string)
        if (child.node.id.indexOf(coreName) === 0) {
            const components = child.node.id.split('-');
            if (components.length !== 2) {
                throw new Error("Incorrectly formatted synthesized construct ID");
            }
            const usedSuffix = Number(components[1]);
            if (usedSuffix > lastSuffix) {
                lastSuffix = usedSuffix;
            }
        }
    });
    return `${coreName}-${lastSuffix + 1}`;
}
function getLambdaVpcSecurityGroupIds(lambdaFunction) {
    const securityGroupIds = [];
    lambdaFunction.connections.securityGroups.forEach(element => securityGroupIds.push(element.securityGroupId));
    return securityGroupIds;
}
exports.getLambdaVpcSecurityGroupIds = getLambdaVpcSecurityGroupIds;
//# sourceMappingURL=data:application/json;base64,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