"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildElasticSearchCWAlarms = exports.buildElasticSearch = void 0;
const elasticsearch = require("aws-cdk-lib/aws-elasticsearch");
const elasticsearch_defaults_1 = require("./elasticsearch-defaults");
const utils_1 = require("./utils");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const MaximumAzsInElasticsearchDomain = 3;
function buildElasticSearch(scope, props) {
    let subnetIds = [];
    const constructDrivenProps = {};
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = createKibanaCognitoRole(scope, props.userpool, props.identitypool, props.domainName);
    if (props.vpc) {
        subnetIds = retrievePrivateSubnetIds(props.vpc);
        if (subnetIds.length > MaximumAzsInElasticsearchDomain) {
            subnetIds = subnetIds.slice(0, MaximumAzsInElasticsearchDomain);
        }
        constructDrivenProps.vpcOptions = {
            subnetIds,
            securityGroupIds: props.securityGroupIds
        };
        // If the client did not submit a ClusterConfig, then we will create one
        if (!props.clientDomainProps?.elasticsearchClusterConfig) {
            constructDrivenProps.elasticsearchClusterConfig = createClusterConfiguration(subnetIds.length);
        }
    }
    else { // No VPC
        // If the client did not submit a ClusterConfig, then we will create one based on the Region
        if (!props.clientDomainProps?.elasticsearchClusterConfig) {
            constructDrivenProps.elasticsearchClusterConfig = createClusterConfiguration(cdk.Stack.of(scope).availabilityZones.length);
        }
    }
    const defaultCfnDomainProps = elasticsearch_defaults_1.DefaultCfnDomainProps(props.domainName, cognitoKibanaConfigureRole, props);
    const finalCfnDomainProps = utils_1.consolidateProps(defaultCfnDomainProps, props.clientDomainProps, constructDrivenProps);
    const esDomain = new elasticsearch.CfnDomain(scope, `ElasticsearchDomain`, finalCfnDomainProps);
    utils_1.addCfnSuppressRules(esDomain, [
        {
            id: "W28",
            reason: `The ES Domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific ES instance only`,
        },
        {
            id: "W90",
            reason: `This is not a rule for the general case, just for specific use cases/industries`,
        },
    ]);
    return [esDomain, cognitoKibanaConfigureRole];
}
exports.buildElasticSearch = buildElasticSearch;
function buildElasticSearchCWAlarms(scope) {
    // Setup CW Alarms for ES
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace',
            statistic: 'Minimum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 20000,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildElasticSearchCWAlarms = buildElasticSearchCWAlarms;
function retrievePrivateSubnetIds(vpc) {
    let targetSubnetType;
    if (vpc.isolatedSubnets.length) {
        targetSubnetType = ec2.SubnetType.PRIVATE_ISOLATED;
    }
    else if (vpc.privateSubnets.length) {
        targetSubnetType = ec2.SubnetType.PRIVATE_WITH_NAT;
    }
    else {
        throw new Error('Error - ElasticSearch Domains can only be deployed in Isolated or Private subnets');
    }
    const subnetSelector = {
        onePerAz: true,
        subnetType: targetSubnetType
    };
    return vpc.selectSubnets(subnetSelector).subnetIds;
}
function createClusterConfiguration(numberOfAzs) {
    return {
        dedicatedMasterEnabled: true,
        dedicatedMasterCount: 3,
        zoneAwarenessEnabled: true,
        zoneAwarenessConfig: {
            availabilityZoneCount: numberOfAzs
        },
        instanceCount: numberOfAzs,
    };
}
function createKibanaCognitoRole(scope, userPool, identitypool, domainName) {
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = new iam.Role(scope, "CognitoKibanaConfigureRole", {
        assumedBy: new iam.ServicePrincipal("es.amazonaws.com"),
    });
    const cognitoKibanaConfigureRolePolicy = new iam.Policy(scope, "CognitoKibanaConfigureRolePolicy", {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateElasticsearchDomainConfig",
                ],
                resources: [
                    userPool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`,
                ],
            }),
            new iam.PolicyStatement({
                actions: ["iam:PassRole"],
                conditions: {
                    StringLike: {
                        "iam:PassedToService": "cognito-identity.amazonaws.com",
                    },
                },
                resources: [cognitoKibanaConfigureRole.roleArn],
            }),
        ],
    });
    cognitoKibanaConfigureRolePolicy.attachToRole(cognitoKibanaConfigureRole);
    return cognitoKibanaConfigureRole;
}
//# sourceMappingURL=data:application/json;base64,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