"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCachePort = exports.createCacheSubnetGroup = exports.obtainMemcachedCluster = void 0;
const cache = require("aws-cdk-lib/aws-elasticache");
const elasticache_defaults_1 = require("./elasticache-defaults");
const utils_1 = require("./utils");
function obtainMemcachedCluster(scope, id, props) {
    if (props.existingCache) {
        props.existingCache.vpcSecurityGroupIds?.push(props.cacheSecurityGroupId);
        return props.existingCache;
    }
    else {
        if (!props.cachePort) {
            throw Error('props.cachePort required for new caches');
        }
        // Create the subnet group from all the isolated subnets in the VPC
        const subnetGroup = createCacheSubnetGroup(scope, props.vpc, id);
        const defaultProps = elasticache_defaults_1.GetMemcachedDefaults(id, props.cachePort);
        const requiredConstructProps = {
            vpcSecurityGroupIds: [props.cacheSecurityGroupId],
            cacheSubnetGroupName: subnetGroup.cacheSubnetGroupName,
        };
        const consolidatedProps = utils_1.consolidateProps(defaultProps, props.cacheProps, requiredConstructProps, true);
        const newCache = new cache.CfnCacheCluster(scope, `${id}-cluster`, consolidatedProps);
        newCache.addDependsOn(subnetGroup);
        return newCache;
    }
}
exports.obtainMemcachedCluster = obtainMemcachedCluster;
function createCacheSubnetGroup(construct, vpc, id) {
    // Memcached has no auth, all access control is
    // network based, so, at least initially, we will
    // only launch it in isolated subnets.
    const subnetIds = [];
    vpc.isolatedSubnets.forEach((subnet) => {
        subnetIds.push(subnet.subnetId);
    });
    return new cache.CfnSubnetGroup(construct, `ec-subnetgroup-${id}`, {
        description: "Solutions Constructs generated Cache Subnet Group",
        subnetIds,
        cacheSubnetGroupName: `${id}-subnet-group`,
    });
}
exports.createCacheSubnetGroup = createCacheSubnetGroup;
function getCachePort(clientCacheProps, existingCache) {
    if (existingCache) {
        return existingCache.attrConfigurationEndpointPort;
    }
    else if (clientCacheProps?.port) {
        return clientCacheProps.port;
    }
    else {
        return elasticache_defaults_1.GetDefaultCachePort();
    }
}
exports.getCachePort = getCachePort;
//# sourceMappingURL=data:application/json;base64,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