"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildLogGroup = void 0;
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const logs = require("aws-cdk-lib/aws-logs");
const utils_1 = require("./utils");
function buildLogGroup(scope, logGroupId, logGroupProps) {
    let _logGroupProps;
    // Override user provided CW LogGroup props with the DefaultLogGroupProps
    _logGroupProps = utils_1.consolidateProps(cloudwatch_log_group_defaults_1.DefaultLogGroupProps(), logGroupProps);
    // Set the LogGroup Id
    const _logGroupId = logGroupId ? logGroupId : 'CloudWatchLogGroup';
    // Create the CW Log Group
    const logGroup = new logs.LogGroup(scope, _logGroupId, _logGroupProps);
    // If required, suppress the Cfn Nag WARNINGS
    if (_logGroupProps.retention === logs.RetentionDays.INFINITE) {
        utils_1.addCfnSuppressRules(logGroup, [
            {
                id: 'W86',
                reason: 'Retention period for CloudWatchLogs LogGroups are set to \'Never Expire\' to preserve customer data indefinitely'
            }
        ]);
    }
    if (!_logGroupProps.encryptionKey) {
        utils_1.addCfnSuppressRules(logGroup, [
            {
                id: 'W84',
                reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
            }
        ]);
    }
    return logGroup;
}
exports.buildLogGroup = buildLogGroup;
//# sourceMappingURL=data:application/json;base64,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