

from typing import TYPE_CHECKING

from ..methods.types.sended_action import SendedAction

from ..enums.sender_action import SenderAction
from ..enums.http_method import HTTPMethod
from ..enums.api_path import ApiPath

from ..connection.base import BaseConnection


if TYPE_CHECKING:
    from ..bot import Bot


class SendAction(BaseConnection):
    
    """
    Класс для отправки действия пользователя (например, индикатора печати) в чат.

    Args:
        bot (Bot): Экземпляр бота для выполнения запроса.
        chat_id (int | None): Идентификатор чата. Если None, действие не отправляется.
        action (SenderAction, optional): Тип действия. По умолчанию SenderAction.TYPING_ON.

    Attributes:
        bot (Bot): Экземпляр бота.
        chat_id (int | None): Идентификатор чата.
        action (SenderAction): Тип действия.
    """
    
    def __init__(
            self,
            bot: 'Bot',
            chat_id: int = None,
            action: SenderAction = SenderAction.TYPING_ON
        ):
            self.bot = bot
            self.chat_id = chat_id
            self.action = action

    async def request(self) -> SendedAction:
        
        """
        Выполняет POST-запрос для отправки действия в указанный чат.

        Returns:
            SendedAction: Результат выполнения запроса.
        """
        
        json = {}

        json['action'] = self.action.value

        return await super().request(
            method=HTTPMethod.POST, 
            path=ApiPath.CHATS + '/' + str(self.chat_id) + ApiPath.ACTIONS,
            model=SendedAction,
            params=self.bot.params,
            json=json
        )