# vi: sw=4 ts=4 et:
"""scoring.py - cMonkey scoring base classes

This file is part of cMonkey Python. Please see README and LICENSE for
more information and licensing details.
"""
LOG_FORMAT = '%(asctime)s %(levelname)-8s %(message)s'

import logging
import os
from datetime import date
import cmonkey.datamatrix as dm
import cmonkey.util as util
import cmonkey.membership as memb
import cmonkey.BSCM as BSCM
import numpy as np
import gc

import cmonkey.database as cm2db

# Python2/Python3 compatibility
try:
    import cPickle as pickle
except ImportError:
    import pickle

try:
    xrange
except NameError:
    xrange = range


# Official keys to access values in the configuration map
KEY_ORGANISM_CODE = 'organism_code'
KEY_NUM_ITERATIONS = 'num_iterations'
KEY_MATRIX_FILENAMES = 'matrix_filenames'
KEY_CACHE_DIR = 'cache_dir'
KEY_SEQUENCE_TYPES = 'sequence_types'
KEY_SEARCH_DISTANCES = 'search_distances'
KEY_SCAN_DISTANCES = 'scan_distances'
KEY_MULTIPROCESSING = 'multiprocessing'
KEY_OUTPUT_DIR = 'output_dir'
KEY_STRING_FILE = 'string_file'


def get_scaling(params, id):
    """returns a scaling function for the given prefix from the configuration parameters"""
    scaling = params[id]
    return util.get_iter_fun(params, prefix + 'scaling', params['num_iterations'])


class RunLog:
    """This is a class that captures information about a particular
    scoring function's behavior in a given iteration. In each iteration,
    a scoring function should log whether it was active and which scaling
    was applied.
    It simply appends log entries to a file to keep I/O and database load
    low.
    """
    def __init__(self, name, session, config_params):
        self.name = name
        self.session = session

    def log(self, iteration, was_active, scaling):
        # Write to database. Note: This can easily lock up sqlite
        # That's why I don't call the commit method here to make
        # it getting called less often
        active_value = 1 if was_active else 0
        entry = cm2db.RunLog(logtype=self.name, iteration=iteration,
                             active=active_value, scaling=scaling)
        self.session.add(entry)


class ScoringFunctionBase:
    """Base class for scoring functions"""

    def __init__(self, id, cmrun):
        """creates a function instance"""
        self.id = id
        self.organism = cmrun.organism()
        self.membership = cmrun.membership()
        self.ratios = cmrun.ratios

        # the cache_result parameter can be used by scoring functions
        # or users to fine-tune the behavior during non-compute operations
        # either recall a previous result from RAM or from a pickled
        # state. In general, setting this to True will be the best, but
        # if your environment has little memory, set this to False
        self.cache_result = True
        self.config_params = cmrun.config_params
        if self.config_params is None:
            raise Exception('NO CONFIG PARAMS !!!')

    def check_requirements(self):
        """Give the scoring module an opportunity to check whether the
        requirements to run are all met"""
        pass

    def run_in_iteration(self, i):
        return self.config_params[self.id]['schedule'](i)

    def pickle_path(self):
        """returns the function-specific pickle-path"""
        return '%s/%s_last.pkl' % (self.config_params['output_dir'], self.id)

    def last_cached(self):
        if self.cache_result and ('cached_result' in dir(self)):
            return self.cached_result
        elif os.path.exists(self.pickle_path()):
            with open(self.pickle_path(), 'rb') as infile:
                return pickle.load(infile)
        else:
            return None

    def store_result(self, result):
        """store the result for later, either by pickling them or caching them"""
        if self.cache_result:
            self.cached_result = result
        else:
            # pickle the result for future use
            logging.debug("pickle result to %s", self.pickle_path())
            with open(self.pickle_path(), 'wb') as outfile:
                pickle.dump(result, outfile)

    def current_score_means(self, result_matrix):
        """This function can be overridden by custom functions to provide their
        own score means. The default version computes the means of the result
        matrix, but feel free to provide your own"""
        if result_matrix is None:
            return 0.0
        else:
            return result_matrix.mean()

    def compute(self, iteration_result, reference_matrix=None):
        """general compute method,
        iteration_result is a dictionary that contains the
        results generated by the scoring functions in the
        current computation.
        the reference_matrix is actually a hack that allows the scoring
        function to normalize its scores to the range of a reference
        score matrix. In the normal case, those would be the gene expression
        row scores"""
        iteration = iteration_result['iteration']

        if self.run_in_iteration(iteration):
            logging.debug("running '%s' in iteration %d with scaling: %f",
                          self.id, iteration, self.scaling(iteration))
            computed_result = self.do_compute(iteration_result, reference_matrix)
            self.store_result(computed_result)
        else:
            computed_result = self.last_cached()

        self.run_log.log(iteration,
                         self.run_in_iteration(iteration),
                         self.scaling(iteration_result['iteration']))
        iteration_result['score_means'][self.id] = self.current_score_means(computed_result)
        return computed_result

    def compute_force(self, iteration_result, reference_matrix=None):
        """enforce computation, regardless of the iteration function"""
        iteration = iteration_result['iteration']
        computed_result = self.do_compute(iteration_result,
                                          reference_matrix)
        with open(self.pickle_path(), 'wb') as outfile:
            pickle.dump(computed_result, outfile)

        self.run_log.log(iteration,
                         self.run_in_iteration(iteration),
                         self.scaling(iteration_result['iteration']))
        iteration_result['score_means'][self.id] = self.current_score_means(computed_result)
        return computed_result

    def do_compute(self, iteration_result, ref_matrix=None):
        """this function is the location of the actual computation, derived scoring
        functions must implement this"""
        raise Exception("implement me")

    def num_clusters(self):
        """returns the number of clusters"""
        return self.membership.num_clusters()

    def gene_names(self):
        """returns the gene names"""
        return self.ratios.row_names

    def rows_for_cluster(self, cluster):
        """returns the rows for the specified cluster"""
        return self.membership.rows_for_cluster(cluster)

    def scaling(self, iteration):
        """returns the quantile normalization scaling for the specified iteration"""
        if 'scaling' in self.config_params[self.id]:
            scaling = self.config_params[self.id]['scaling']
            if scaling[0] == 'scaling_const':
                return scaling[1]
            elif scaling[0] == 'scaling_rvec':
                num_iterations = self.config_params['num_iterations']
                return util.get_rvec_fun(scaling[1].replace('num_iterations',
                                                            str(num_iterations)))(iteration)
            else:
                raise Exception("Unknown scaling: '%s'" % scaling[0])
        else:
            return 0.0

    def run_logs(self):
        """returns a list of RunLog objects, giving information about
        the last run of this function"""
        return []


class ColumnScoringFunction(ScoringFunctionBase):
    """Scoring algorithm for microarray data based on conditions.
    Note that the score does not correspond to the normal scoring
    function output format and can therefore not be combined in
    a generic way (the format is |condition x cluster|)"""

    def __init__(self, function_id, cmrun):
        """create scoring function instance"""
        ScoringFunctionBase.__init__(self, function_id, cmrun)

        #BSCM.  Danziger et al. 2015
        self.BSCM_obj = None
        if cmrun.config_params['use_BSCM']:
            # How to pass verbose and so on? More parameters?
            self.BSCM_obj = BSCM.BSCM(self.ratios, verbose=False,
                                      useChi2=cmrun.config_params['use_chi2'])
            #Note: Ratios normalized upstream during loading by config.py module
        self.run_log = RunLog(function_id, cmrun.dbsession(), cmrun.config_params)

    def do_compute(self, iteration_result, ref_matrix=None):
        """compute method, iteration is the 0-based iteration number"""
        return compute_column_scores(self.membership, self.ratios,
                                     self.num_clusters(), self.config_params,
                                     self.BSCM_obj)

    def get_BSCM(self):
        """Return the background sampled coherence matrix object"""
        return self.BSCM_obj


def compute_column_scores(membership, matrix, num_clusters,
                          config_params, BSCM_obj=None):
    """Computes the column scores for the specified number of clusters"""

    def compute_substitution(cluster_column_scores):
        """calculate substitution value for missing column scores"""
        membership_values = []
        for cluster in xrange(1, num_clusters + 1):
            columns = membership.columns_for_cluster(cluster)
            column_scores = cluster_column_scores[cluster - 1]
            if column_scores is not None:
                colnames, scores = column_scores
                for col in xrange(len(colnames)):
                    if colnames[col] in columns:
                        membership_values.append(scores[col])
        return util.quantile(membership_values, 0.95)

    def make_submatrix(cluster):
        row_names = membership.rows_for_cluster(cluster)
        if len(row_names) > 1:
            return matrix.submatrix_by_name(row_names=row_names)
        else:
            return None

    cluster_column_scores = [] #To be filled or overwritten
    if BSCM_obj is None:
        if config_params['multiprocessing']:
            with util.get_mp_pool(config_params) as pool:
                    cluster_column_scores = pool.map(compute_column_scores_submatrix, map(make_submatrix, xrange(1, num_clusters + 1)))
        else:
            for cluster in xrange(1, num_clusters + 1):
                cluster_column_scores.append(compute_column_scores_submatrix(
                    make_submatrix(cluster)))
    else: #if BSCM_obj exists
        num_cores = 1
        if not config_params['num_cores'] is None:
            num_cores = config_params['num_cores']

        for cluster in xrange(1, num_clusters + 1):
            if make_submatrix(cluster) is None:
                cluster_column_scores.append(None)
            else:
                cur_column_scores = BSCM_obj.getPvals(make_submatrix(cluster).row_names, num_cores=num_cores)
                exp_names = list(cur_column_scores.keys())
                exp_scores = np.array(list(cur_column_scores.values()))
                cluster_column_scores.append((exp_names, exp_scores))

    substitution = compute_substitution(cluster_column_scores)

    # Convert scores into a matrix that have the clusters as columns
    # and conditions in the rows
    result = dm.DataMatrix(matrix.num_columns, num_clusters,
                           row_names=matrix.column_names)
    rvalues = result.values
    for cluster in xrange(num_clusters):
        column_scores = cluster_column_scores[cluster]

        if column_scores is not None:
            _, scores = column_scores
            scores[np.isnan(scores)] = substitution

        for row_index in xrange(matrix.num_columns):
            if column_scores is None:
                rvalues[row_index, cluster] = substitution
            else:
                _, scores = column_scores
                rvalues[row_index, cluster] = scores[row_index]
    result.fix_extreme_values()
    return result


def compute_column_scores_submatrix(matrix):
    """For a given matrix, compute the column scores.
    This is used to compute the column scores of the sub matrices that
    were determined by the pre-seeding, so typically, matrix is a
    submatrix of the input matrix that contains only the rows that
    belong to a certain cluster.
    The result is a DataMatrix with one row containing all the
    column scores

    This function normalizes diff^2 by the mean expression level, similar
    to "Index of Dispersion", see
    http://en.wikipedia.org/wiki/Index_of_dispersion
    for details
    """
    if matrix is None:
        return None
    colmeans = util.column_means(matrix.values)
    matrix_minus_colmeans_squared = np.square(matrix.values - colmeans)
    var_norm = np.abs(colmeans) + 0.01
    result = util.column_means(matrix_minus_colmeans_squared) / var_norm
    return (matrix.column_names, result)


def combine(result_matrices, score_scalings, membership, iteration, config_params):
    """This is  the combining function, taking n result matrices and scalings"""
    quantile_normalize = config_params['quantile_normalize']

    for i, m in enumerate(result_matrices):
        m.fix_extreme_values()
        m.subtract_with_quantile(0.99)

        # debug mode: print scoring matrices before combining
        if ('dump_scores' in config_params['debug'] and
            (iteration == 1 or (iteration % config_params['debug_freq'] == 0))):
            funs = config_params['pipeline']['row-scoring']['args']['functions']
            m.write_tsv_file(os.path.join(config_params['output_dir'], 'score-%s-%04d.tsv' % (funs[i]['id'], iteration)), compressed=False)

    if quantile_normalize:
        if len(result_matrices) > 1:
            start_time = util.current_millis()
            result_matrices = dm.quantile_normalize_scores(result_matrices,
                                                           score_scalings)
            elapsed = util.current_millis() - start_time
            logging.debug("quantile normalize in %f s.", elapsed / 1000.0)

        in_matrices = [m.values for m in result_matrices]

    else:
        in_matrices = []
        num_clusters = membership.num_clusters()
        mat = result_matrices[0]
        index_map = {name: index for index, name in enumerate(mat.row_names)}
        # we assume matrix 0 is always the gene expression score
        # we also assume that the matrices are already extreme value
        # fixed
        rsm = []
        for cluster in range(1, num_clusters + 1):
            row_members = sorted(membership.rows_for_cluster(cluster))
            rsm.extend([mat.values[index_map[row], cluster - 1] for row in row_members])
        scale = util.mad(rsm)
        if scale == 0:  # avoid that we are dividing by 0
            scale = util.r_stddev(rsm)
        if scale != 0:
            median_rsm = util.median(rsm)
            rsvalues = (mat.values - median_rsm) / scale
            num_rows, num_cols = rsvalues.shape
            rscores = dm.DataMatrix(num_rows, num_cols,
                                    mat.row_names,
                                    mat.column_names,
                                    values=rsvalues)
            rscores.fix_extreme_values()
        else:
            logging.warn("combiner scaling -> scale == 0 !!!")
            rscores = mat
        in_matrices.append(rscores.values)

        if len(result_matrices) > 1:
            rs_quant = util.quantile(rscores.values, 0.01)
            logging.debug("RS_QUANT = %f", rs_quant)
            for i in range(1, len(result_matrices)):
                values = result_matrices[i].values
                qqq = abs(util.quantile(values, 0.01))
                if qqq == 0:
                    logging.debug('SPARSE SCORES - %d attempt 1: pick from sorted values', i)
                    qqq = sorted(values.ravel())[9]
                if qqq == 0:
                    logging.debug('SPARSE SCORES - %d attempt 2: pick minimum value', i)
                    qqq = abs(values.min())
                if qqq != 0:
                    values = values / qqq * abs(rs_quant)
                else:
                    logging.debug('SPARSE SCORES - %d not normalizing!', i)
                in_matrices.append(values)

    if len(result_matrices) > 0:
        start_time = util.current_millis()
        # assuming same format of all matrices
        combined_score = np.zeros(in_matrices[0].shape)
        for i in xrange(len(in_matrices)):
            combined_score += in_matrices[i] * score_scalings[i]

        elapsed = util.current_millis() - start_time
        logging.debug("combined score in %f s.", elapsed / 1000.0)
        matrix0 = result_matrices[0]  # as reference for names
        return dm.DataMatrix(matrix0.num_rows, matrix0.num_columns,
                             matrix0.row_names, matrix0.column_names,
                             values=combined_score)
    else:
        return None


class ScoringFunctionCombiner:
    """Taking advantage of the composite pattern, this combiner function
    exposes the basic interface of a scoring function in order to
    allow for nested scoring functions as they are used in the motif
    scoring
    """
    def __init__(self, organism, membership, scoring_functions, config_params=None):
        """creates a combiner instance"""
        self.organism = organism  # not used, but constructor interface should be the same
        self.membership = membership
        self.scoring_functions = scoring_functions
        self.config_params = config_params

    def check_requirements(self):
        """Give the scoring module an opportunity to check whether the
        requirements to run are all met"""
        for fun in self.scoring_functions:
            fun.check_requirements()

    def compute_force(self, iteration_result, ref_matrix=None):
        """compute scores for one iteration, recursive force"""
        result_matrices = []
        score_scalings = []
        reference_matrix = ref_matrix
        iteration = iteration_result['iteration']
        for scoring_function in self.scoring_functions:
            # clean up before doing something complicated
            gc.collect()

            if reference_matrix is None and len(result_matrices) > 0:
                reference_matrix = result_matrices[0]

            matrix = scoring_function.compute_force(iteration_result, reference_matrix)
            if matrix is not None:
                result_matrices.append(matrix)
                score_scalings.append(scoring_function.scaling(iteration))

                if self.config_params['log_subresults']:
                    self.log_subresult(scoring_function, matrix)
        return combine(result_matrices, score_scalings, self.membership,
                       iteration, self.config_params)

    def compute(self, iteration_result, ref_matrix=None):
        """compute scores for one iteration"""
        result_matrices = []
        score_scalings = []
        reference_matrix = ref_matrix
        iteration = iteration_result['iteration']
        for scoring_function in self.scoring_functions:
            # clean up before doing something complicated
            gc.collect()

            # This  is actually a hack in order to propagate
            # a reference matrix to the compute function
            # This could have negative impact on scalability
            if reference_matrix is None and len(result_matrices) > 0:
                reference_matrix = result_matrices[0]

            matrix = scoring_function.compute(iteration_result, reference_matrix)
            if matrix is not None:
                result_matrices.append(matrix)
                score_scalings.append(scoring_function.scaling(iteration))

                if self.config_params['log_subresults']:
                    self.log_subresult(scoring_function, matrix)

        return combine(result_matrices, score_scalings, self.membership,
                       iteration, self.config_params)

    def combine_cached(self, iteration):
        """Combine the cached results of the contained scoring function.
        This is used by the post adjustment"""
        result_matrices = []
        score_scalings = []
        for scoring_function in self.scoring_functions:
            matrix = scoring_function.last_cached()
            if matrix is not None:
                result_matrices.append(matrix)
                score_scalings.append(scoring_function.scaling(iteration))

        return combine(result_matrices, score_scalings, self.membership,
                       iteration, self.config_params)


    def log_subresult(self, score_function, matrix):
        """output an accumulated subresult to the log"""
        scores = []
        mvalues = matrix.values
        for cluster in xrange(1, matrix.num_columns + 1):
            cluster_rows = self.membership.rows_for_cluster(cluster)
            for row in xrange(matrix.num_rows):
                if matrix.row_names[row] in cluster_rows:
                    scores.append(mvalues[row][cluster - 1])
        logging.debug("function '%s', trim mean score: %f",
                      score_function.id,
                      util.trim_mean(scores, 0.05))

    def scaling(self, iteration):
        """returns the scaling for the specified iteration"""
        return self.scaling_func(iteration)

    def run_logs(self):
        """joins all contained function's run logs"""
        result = []
        for scoring_func in self.scoring_functions:
            result.extend(scoring_func.run_logs())
        return result

__all__ = ["ColumnScoringFunction"]
