"""mast.py - MAST format readers
Handles the file format for MAST for versions >= 4.3.0

This file is part of cm2meme. Please see README and LICENSE for
more information and licensing details.
"""
import xml.etree.ElementTree as ET
import re

import cmonkey.meme.util as util

try:
    xrange
except NameError:
    xrange = range


def from_xml_text(output_text, genes):
    """Reads p/e values and gene annotations from a string in
    in MAST XML format.
    Inputs: - output_text: a string in MAST XML output format
    ------- - genes: a list of genes that were used as input to
              the previous MEME run
    Returns: a pair (pevalues, annotations)
    -------- - pevalues is [(gene, pval, eval)]
             - annotations is a dictionary gene -> [(pval, pos, motifnum)]"""
    if output_text is None:  # there was an error in mast, ignore its output
        return [], {}

    root = ET.fromstring(output_text)
    version = list(map(int, root.get('version').split('.')))
    # to match the schema changes
    major, minor, patch = version
    use_411 = True
    if major == 4 and minor < 11:
        use_411 = False
    if use_411:
        return __read_mast_output_xml_4_11(root, genes, version)
    else:
        return __read_mast_output_xml_4_8(root, genes)


def __read_mast_output_xml_4_8(root, genes):
    pevalues = []
    annotations = {}
    for sequence in root.iter('sequence'):
        score = sequence.find('score')
        seqname = sequence.get('name')
        if not seqname in annotations:
            annotations[seqname] = []
        pevalues.append((seqname,
                         float(score.get('combined_pvalue')),
                         float(score.get('evalue'))))
        if seqname in genes:
            for hit in sequence.iter('hit'):
                strand = hit.get('strand')
                motifnum = int(hit.get('motif').replace('motif_', ''))
                if strand == 'reverse':
                    motifnum = -motifnum
                annot = (float(hit.get('pvalue')),
                         int(hit.get('pos')) + 2,  # like R cmonkey
                         motifnum)
                annotations[seqname].append(annot)
    return pevalues, annotations


def __read_mast_output_xml_4_11(root, genes, version):
    """Starting from 4.11.x, MAST generates a different output format"""
    major, minor, patch = version
    pevalues = []
    annotations = {}

    # motif numbering changes starting in 4.11.4:
    # 1. the id field changes to the consensus string
    # 2. the alt field changes to "MEME-<motif number>"
    if major == 4 and minor == 11 and patch <= 3:
        motif_nums = [int(motif.get('id')) for motif in root.iter('motif')]
    else:
        motif_nums = [int(motif.get('alt').replace('MEME-', '')) for motif in root.iter('motif')]

    for sequence in root.iter('sequence'):
        score = sequence.find('score')
        seqname = sequence.get('name')
        if not seqname in annotations:
            annotations[seqname] = []
        pevalues.append((seqname,
                         float(score.get('combined_pvalue')),
                         float(score.get('evalue'))))
        if seqname in genes:
            for hit in sequence.iter('hit'):
                strand = hit.get('strand')
                # the motif number is now encoded in the motif database list
                motifnum = motif_nums[int(hit.get('idx'))]
                if strand == 'reverse':
                    motifnum = -motifnum
                annot = (float(hit.get('pvalue')),
                         int(hit.get('pos')) + 2,  # like R cmonkey
                         motifnum)
                annotations[seqname].append(annot)
    return pevalues, annotations


def from_430_text(output_text, genes):
    """Reads out the p-values and e-values and the gene annotations
    from a mast output string. This format is generated by
    MAST 4.30 and is only here to support the legacy format.
    Use the XML version instead, it is more reliable.
    """
    def next_pe_value_line(start_index, lines):
        """Find the next combined p-value and e-value line"""
        return util.next_regex_index('.*COMBINED P-VALUE.*',
                                     start_index, lines)

    def read_pe_values(lines):
        """read all combined p-values and e-values"""
        result = []
        current_index = next_pe_value_line(0, lines)
        while current_index != -1:
            gene = lines[current_index - 2].strip()
            line = lines[current_index]
            pvalue = float(util.extract_regex('P-VALUE\s+=\s+(\S+)', line))
            evalue = float(util.extract_regex('E-VALUE\s+=\s+(\S+)', line))
            result.append((gene, pvalue, evalue))
            current_index = next_pe_value_line(current_index + 1, lines)
        return result

    def read_seqalign_blocks(lines, start_index, seqlen):
        """Read the sequence alignment blocks starting at start_index
        a block has the format:
        1. motif number line (+/- = forward/reverse)
        2. pvalue line
        3. motif sequence line
        4. alignment/match line
        5. gene sequence line
        6. blank line (separator)
        -> Repeat this pattern until the whole database sequence printed

        While the mast output is easily human-readable, it
        is hard to parse programmatically.
        This method does it as follows:
        - read all motif numbers in sequence
        - read all p-values in sequencs
        - the motif number opening brackets are regarded as position markers

        for each block, we only need to keep track in which column the gene
        sequence starts and at which relative position we are
        """
        current_index = start_index
        is_last = False

        # global lines
        motifnum_line = ""
        pvalue_line = ""
        seq_line = ""
        while not is_last:
            is_last = is_last_block(lines, current_index, seqlen)
            # append to the motif number line, the p-value line, and seq line
            motifnum_line += lines[current_index].rstrip().ljust(80)[5:]
            pvalue_line += lines[current_index + 1].rstrip().ljust(80)[5:]
            seq_line += lines[current_index + 4].rstrip().ljust(80)[5:]
            current_index += 6

        motif_nums = read_motif_numbers(motifnum_line)
        positions = read_positions(motifnum_line, seq_line)
        pvalues = read_pvalues(pvalue_line, [(pos - 2) for pos in positions])
        return zip(pvalues, positions, motif_nums)

    def read_motifnum_line(line):
        """format and pad a motif number line"""
        return line.rstrip().ljust(80)[5:]

    def is_last_block(lines, index, seqlen):
        """determines whether the specified block is the last one for
        the current gene"""
        seqline = None
        try:
            seqline = lines[index + 4]
            seqstart_index = int(re.match('(\d+).*', seqline).group(1))
            seq_start = re.match('\d+\s+(\S+)', seqline).start(1)
            return ((len(seqline) - seq_start) + seqstart_index >= seqlen or
                    not re.match('(\d+).*', lines[index + 10]))
        except:
            if seqline is not None:
                print("ERROR IN SEQLINE: [%s]" % seqline)

    def read_motif_numbers(motifnum_line):
        """reads the motif numbers contained in a motif number line"""
        return [int(re.sub('\[|\]', '', motifnum))
                for motifnum in re.split(' +', motifnum_line)
                if len(motifnum.strip()) > 0]

    def read_pvalues(pvalue_line, indexes):
        """reads the p-values contained in a p-value line"""
        def make_float(s):
          """unfortunately, MEME result lines can have weird float formats"""
          return float(s.replace(' ', ''))
        pvalues = []
        for index_num in xrange(len(indexes)):
            if index_num < len(indexes) - 1:
                pvalues.append(
                    make_float(pvalue_line[indexes[index_num]:
                                      indexes[index_num + 1]]))
            else:
                pvalues.append(make_float(pvalue_line[indexes[index_num]:]))
        return pvalues

    def read_positions(motifnum_line, seqline):
        """we only need the motif number line and the sequence line
        to retrieve the position"""
        # offset +2 for compatibility with cMonkey R, don't really
        # know why we need this
        try:
            return [(m.start() + 2)
                    for m in re.finditer('\[', motifnum_line)]
        except:
            logging.error("ERROR in read_positions(), motifnum_line: '%s'",
                          str(motifnum_line))

    def read_annotations(lines, genes):
        """extract annotations, genes are given as refseq ids"""
        result = {}
        current_index = next_pe_value_line(0, lines)
        while current_index != -1:
            gene = lines[current_index - 2].strip()
            if gene in genes:
                info_line = lines[current_index]
                length = int(util.extract_regex('LENGTH\s+=\s+(\d+)', info_line))
                has_seqalign_block = True
                diagram_match = re.match('^\s+DIAGRAM:\s+(\d+)$',
                                         lines[current_index + 1])
                if diagram_match is not None:
                    diagram = int(diagram_match.group(1))
                    if diagram == length:
                        has_seqalign_block = False

                if has_seqalign_block:
                    # the diagram line can span several lines and the blank
                    # line after those can span several, so search for the
                    # first non-blank line after the block of blank lines
                    blank_index = current_index + 2
                    while len(lines[blank_index].strip()) > 0:
                        blank_index += 1
                    non_blank_index = blank_index + 1
                    while len(lines[non_blank_index].strip()) == 0:
                        non_blank_index += 1
                    result[gene] = read_seqalign_blocks(lines,
                                                        non_blank_index,
                                                        length)

            current_index = next_pe_value_line(current_index + 1, lines)
        return result

    # Make sure MAST returns a meaningful result
    if output_text.startswith("Error reading log-odds matrix file"):
        logging.warn("MAST returned the famous 'Error reading log-odds " +
                     "matrix file, provide empty result...'")
        return ([], {})
    else:
        lines = output_text.split('\n')
        pe_values = read_pe_values(lines)
        annotations = read_annotations(lines, genes)
        return (pe_values, annotations)
