# -*- coding: utf-8 -*-
"""MX RT class file."""

import os
import re
import sys


class RT(object):
    """RT class."""

    def __init__(self, update):
        """Initialize a class instance."""
        self.update = update
        self.mx = update.mx
        self.verbose = update.mx.verbose

        # define the name of the rt alias file
        self.aliasfile = 'aliases.rt'

        # define the path to the rt puppet file
        self.puppetfile = '%s/%s%s' % (
            self.mx.aliases_puppetdir,
            self.aliasfile,
            self.mx.extension,
        )

        # get program name
        self.program = os.path.basename(sys.argv[0])

        # server hosts
        self.help_host = 'help.broadinstitute.org'
        self.rt_host = 'rt.broadinstitute.org'

        # bitsdb monogo collection
        self.collection = {}

    def generate(self):
        """Return the contents of the puppetfile."""
        # define header
        output = '# This file is autogenerated by the bits-mx %s script.\n' % (
            self.program
        )
        output += '# Do not make local modifications to this file.\n'

        # get aliases
        help_aliases = self.get_aliases(self.update.help_queues, self.help_host)
        rt_aliases = self.get_aliases(self.update.rt_queues, self.rt_host)

        # users without a forward
        output += '\n#\n# %s aliases\n#\n' % (self.help_host)
        output += '\n'.join(help_aliases)

        # users with a forward
        output += '\n\n#\n# %s aliases\n#\n' % (self.rt_host)
        output += '\n'.join(rt_aliases)
        return output

    def get_alias(self, queue, mailhost):
        """Return an alias entry for an RT queue."""
        # get name and description
        name = queue['Name']
        description = queue['Description']

        # get correspond and comment address
        correspond = queue['CorrespondAddress'].lower()
        comment = queue['CommentAddress'].lower()

        # skip non-broad correspond and comment addresses
        if not correspond or not re.search('@broadinstitute.org', correspond):
            return
        if not comment or not re.search('@broadinstitute.org', comment):
            return

        # process correspond address
        correspond_username = correspond.split('@')[0]
        data = {
            'name': name,
            'description': description,
            'email': correspond,
            'username': correspond_username,
            'type': 'CorrespondAddress',
        }
        if correspond not in self.collection:
            # store for mongo
            self.collection[correspond] = data
        else:
            print('ERROR: Duplicate correspond email: %s' % (correspond))

        # process comment address
        comment_username = comment.split('@')[0]
        data = {
            'name': name,
            'description': description,
            'email': comment,
            'username': comment_username,
            'type': 'CommentAddress',
        }
        if comment not in self.collection:
            # store for mongo
            self.collection[comment] = data
        else:
            print('ERROR: Duplicate comment email: %s' % (comment))

        # create alias header entry
        header = '\n# %s - %s' % (name, description)

        # create correspond entry
        correspond_entry = '%s: %s@%s' % (correspond_username, correspond_username, mailhost)

        # create comment entry
        comment_entry = '%s: %s@%s' % (comment_username, comment_username, mailhost)

        # create entry
        entry = [header, correspond_entry, comment_entry]
        return '\n'.join(entry)

    def get_aliases(self, queues, mailhost):
        """Return the rules with forwards."""
        aliases = []
        for qid in sorted(queues, key=lambda x: queues[x]['Name'].lower()):
            queue = queues[qid]
            # skip disabled queues
            if queue['Disabled']:
                continue
            entry = self.get_alias(queue, mailhost)
            # skip empty entries
            if not entry:
                continue
            # add entry
            aliases.append(entry)

        return aliases
