"""
This module contains definitions of classes to represent ONLY additional data from the CIM import,
which is not part of the original Matpower data structure, but might be necessary for the GARPUR project.

.. note:: The data in the classes in this file is structured according to how it will be represented in Matlab, and not
   according to how it will be represented in the Python import code.

.. note:: The current implementation refers to ``CIMv14`` !

.. seealso::  :py:mod:`Topology_BusBranch` and :py:mod:`CIM2Matpower`

:Date: 2016-05-10
:Authors: Konstantin Gerasimov
:e-mail: kkgerasimov@gmail.com
:Credits: This function is created for KU-Leuven as part of the GARPUR project http://www.garpur-project.eu
"""

from itertools import count


class Node(object):
    """Repsenets the basic info about a ``IEC61970::Topology::TopologicalNode`` object from the source CIM file."""

    id_generator = count(1)

    def __init__(self, id_cim, bus_id=-1, name='', desc=''):
        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.bus_id = bus_id
        """A :py:attr:`Topology_BusBranch.Bus.id` (integer)"""

        self.name = name
        """The ``name`` attribute (string)"""

        self.desc = desc
        """The ``description`` attribute (string)"""

        self.id = next(self.id_generator)
        """An autogenerated ID, subsequent numbering, starting from 1, (integer)"""


class Substation(object):
    """
    Repsents the basic info about a ``IEC61970::Core::Substation`` object from the source CIM file.
    """

    def __init__(self, node_id_list, id_cim, name=''):
        self.node_id_list = node_id_list
        """A list of :py:attr:`Topology_NodeBreaker.Node.id``-s within the substation (integers)"""

        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.name = name
        """The ``name`` attribute (string)"""


class Area(object):
    """Repsenets the basic info about a ``IEC61970::ControlArea::ControlArea`` object from the source CIM file."""

    id_generator = count(1)

    def __init__(self, id_cim, name='', netInterchange=0, pTolerance=0):
        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.name = name
        """The ``name`` attribute (string)"""

        self.netInterchange = netInterchange
        """The ``netInterchange`` attribute (ActivePower datatype, usually MW for TSO networks)"""

        self.pTolerance = pTolerance
        """The ``pTolerance`` attribute (ActivePower datatype, usually MW for TSO networks)"""

        self.id = next(self.id_generator)
        """
        An autogenerated ID, subsequent numbering, starting from 1, (integer)

        .. note:: This ID goes in the Matpower ``mpc.bus(:,BUS_AREA)`` struct field.
        """


class Zone(object):
    """
    Repsenets the basic info about a ``IEC61970::LoadModel::LoadArea`` object from the source CIM file.

    .. warning:: The current implementation of the :py:meth:`CIM2Matpower.CIM2Matpower._iterate_prim_eq` is fine tuned
        for the RTE CIM data in which there is no ``IEC61970::LoadModel::LoadArea`` objects and because of that instead
        the ``IEC61970::Core::SubGeographicalRegion`` objects are treated as load *Zone*-s.
    """

    id_generator = count(1)

    def __init__(self, id_cim, name=''):
        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.name = name
        """The ``name`` attribute (string)"""

        self.id = next(self.id_generator)
        """
        An autogenerated ID, subsequent numbering, starting from 1, (integer)

        .. note:: This ID goes in the Matpower ``mpc.bus(:,ZONE) struct field.``
        """


class Switch(object):
    """Repsenets the basic info about a ``IEC61970::Wires::Switch`` object from the source CIM file."""

    def __init__(self, node_from_id, node_to_id, id_cim, name='', status=False, branch_id=0):
        self.node_from_id = node_from_id
        """A :py:attr:`Topology_NodeBreaker.Node.id` on Terminal 0 (integer)"""

        self.node_to_id = node_to_id
        """A :py:attr:`Topology_NodeBreaker.Node.id` on Terminal 1 (integer)"""

        self.id_cim = id_cim
        """The ``name`` attribute (string)"""

        self.name = name
        """The ``name`` attribute (string)"""

        self.status = status
        """ The *Swtich* status (Open or Closed), i.e. False if either of the *Swtich* terminal's attribute ``connected`` is False. (boolean)"""

        self.branch_id = branch_id
        """
        A :py:attr:`Topology_BusBranch.Branch.id` (integer)

        .. note:: This property only reserves place. The Matpower case, which comes out of this import tool is a *BusBranch* model,
            therefore it should not have branches representing *Switch*-es. However, it the created in Matlab function ``mpc_bb2nodal.m``
            is called, it creates a NodeBreaker model by adding the *Switch*-es as low impedence *Branch*-es and stores the
            :py:attr:`Topology_BusBranch.Branch.id` for the opposite transformation. This information is also necessary
            when switching actions are to be performed.
        """


class Branch(object):  # cim_branches list
    """
    Represents additional to the :py:class:`Topology_BusBranch.Branch` data about either ``IEC61970::Wires::ACLineSegment``
    or ``IEC61970::Wires::PowerTransformer`` object from the source CIM file, which formed that *Branch*.
    """

    id_generator = count(1)

    def __init__(self, node_from_id, node_to_id, id_cim, name='', status_from=False, status_to=False):
        self.node_from_id = node_from_id
        """A :py:attr:`Topology_NodeBreaker.Node.id`` on Terminal 0 (integer)"""

        self.node_to_id = node_to_id
        """A :py:attr:`Topology_NodeBreaker.Node.id`` on Terminal 1 (integer)"""

        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.name = name
        """The ``name`` attribute (string)"""

        self.status_from = status_from
        """
        The *Branch* FROM terminal status (Open or Closed), i.e. terminal's attribute ``connected``. (boolean)

        .. note:: For ``ACLineSegment``-s the FROM terminal is terminal 0.
            For ``PowerTransformer``-s however it is the **secondary** winding's terminal (in order to comply with the
            Matpower transformer modeling assumptions)!

        """

        self.status_to = status_to
        """
            The *Branch* TO terminal status (Open or Closed), i.e. terminal's attribute ``connected``. (boolean)

            .. note:: For ``ACLineSegment``-s the TO terminal is terminal 1.
                For ``PowerTransformer``-s however it is the **primary** winding's terminal (in order to comply with the
                Matpower transformer modeling assumptions)!
        """

        self.id = next(self.id_generator)
        """Autogenerated ID, subsequent numbering, starting from 1, (integer)"""


class Generator(object):
    """
    Repsenets the basic info about a ``IEC61970::Wires::SynchronousMachine`` object from the source CIM file,
    not contained in the Matpower ``mpc.gen`` field.
    """

    id_generator = count(1)

    def __init__(self, node_id, id_cim, type, name='', mode='', fuel_type=''):
        self.node_id = node_id
        """A :py:attr:`Topology_NodeBreaker.Node.id`` to which the *Generator* is connected (integer)"""

        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.name = name
        """The ``name`` attribute (string)"""

        self.mode = mode
        """The ``operatingMode`` attribute (string)"""

        self.type = type
        """
        The prefix of the ``GeneratingUnit`` class reference (string)

        .. warning:: Some *Generator*-s are assigned to the generic ``GeneratingUnit`` class. Then their *type* is an empty sting!
        """

        self.fuel_type = fuel_type
        """The ``FossilFuel.fossilFuelType`` attribute (string) of the ``ThermalGeneratingUnit`` CIM class"""

        self.id = next(self.id_generator)
        """Autogenerated ID, subsequent numbering, starting from 1, (integer)"""


class Load(object):
    """
    Repsenets the basic info about a ``IEC61970::Wires::EnergyConsumer`` object from the source CIM file.

    .. note:: The problem is that the Matpower mpc structure aggeregates all *Load*-s in the same bus. However if is
        useful to preserve the information about them (for example for load shedding) and this is why it
        is stored separately in this class.
    """

    def __init__(self, node_id, id_cim, name='', p_mw=0, q_mvar=0, status=False):
        self.node_id = node_id
        """A :py:attr:`Topology_NodeBreaker.Node.id`` to which the *Load* is connected (integer)"""

        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.name = name
        """The ``name`` attribute (string)"""

        self.p_mw = p_mw
        """The load terminal's StateVariable ``SvPowerFlow.p`` attribute (ActivePower datatype, usually MW for TSO networks)"""

        self.q_mvar = q_mvar
        """The load terminal's StateVariable ``SvPowerFlow.q`` attribute (ReactivePower datatype, usually MVAr for TSO networks)"""

        self.status = status
        """The *Load* terminal's attribute ``connected`` (boolean)"""


class Shunt(object):  # cim_shunts
    """
    Repsenets the basic info about a ``IEC61970::Wires::ShuntCompensator`` object from the source CIM file.

    .. note:: The problem is that the Matpower mpc structure aggeregates all *Shunt*-s in the same bus. However if is
        useful to preserve the information about them (for example for switching actions) and this is why it
        is stored separately in this class.

    .. warning:: ``IEC61970::Wires::SeriesCompensator`` and ``IEC61970::Wires::StaticVarCompensator`` are not yet
        implemented! See the source code of :py:meth:`CIM2Matpower.CIM2Matpower._create_buses_and_gens` !
    """

    def __init__(self, node_id, id_cim, maximumSections, numActiveSections, gPerSection_MVAr, bPerSection_MVAr, name='', status=False):
        self.node_id = node_id
        """A :py:attr:`Topology_NodeBreaker.Node.id`` to which the *Shunt* is connected (integer)"""

        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.name = name
        """The ``name`` attribute (string)"""

        self.status = status
        """The *Shunt* terminal's attribute ``connected`` (boolean)"""

        self.gPerSection_MVAr = gPerSection_MVAr
        """The ``gPerSection`` attribute, transformed MW demanded at V = 1.0 p.u. as required by Matpower specifications"""

        self.bPerSection_MVAr = bPerSection_MVAr
        """The ``bPerSection`` attribute, transformed MVAr injected at V = 1.0 p.u. as required by Matpower specifications"""

        self.maximumSections = maximumSections
        """The ``maximumSections`` attribute (integer)"""

        self.numActiveSections = numActiveSections
        """The *Shunt*'s StateVariable ``SvShuntCompensatorSections.continuousSections`` attribute, i.e. the number of switched ON sections (integer)"""


class PhaseTapChanger(object):
    """Repsenets the basic info about a ``IEC61970::Wires::PhaseTapChanger`` object from the source CIM file."""

    def __init__(self, branch_id, id_cim, step_num_list, x_pu_list, angle_shift_deg_list,
                 continuous_position):
        self.branch_id = branch_id
        """A :py:attr:`Topology_BusBranch.Branch.id`` (integer)"""

        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.step_num_list = step_num_list
        """A list of the step numbers (list of subsequent string)"""

        self.x_pu_list = x_pu_list
        """A list of reactances [p.u.] for each corresponding step  (list of floats)"""

        self.angle_shift_deg_list = angle_shift_deg_list
        """A list of phase shifts [deg] for each corresponding step (list of floats)"""

        self.continuous_position = continuous_position
        """The *PhaseTapChanger*'s StateVariable ``SvTapStep.continuousPosition`` attribute, i.e. the actual step position (integer)"""

class RatioTapChanger(object):
    """Repsenets the basic info about a ``IEC61970::Wires::RatioTapChanger`` object from the source CIM file."""

    def __init__(self, branch_id, id_cim, lowStep, highStep, neutralU_kV, stepVoltageIncrement_kV, continuousPosition,
                 hasRegulatingControl, RC_discrete=[], RC_mode='', RC_targetRange_kV=[], RC_targetValue_kV=[]):
        self.branch_id = branch_id
        """A :py:attr:`Topology_BusBranch.Branch.id`` (integer)"""

        self.id_cim = id_cim
        """The unique ``mRID`` object identifier attribute (string)"""

        self.lowStep = lowStep
        """The ``lowStep`` attribute (inherited from: ``TapChanger``), i.e. the low step position number (integer)"""

        self.highStep = highStep
        """The ``highStep`` attribute (inherited from: ``TapChanger``), i.e. the highest possible tap step position, advance from neutral (integer)"""

        self.neutralU_kV = neutralU_kV
        """The ``neutralU`` attribute (inherited from: ``TapChanger``), i.e. voltage at which the winding operates at the neutral tap setting (kV)"""

        self.stepVoltageIncrement_kV = stepVoltageIncrement_kV
        """The ``stepVoltageIncrement`` attribute (inherited from: ``TapChanger``), i.e. tap step voltage increment, per step position. (kV)"""

        self.continuousPosition = continuousPosition
        """The *RatioTapChanger*'s StateVariable ``SvTapStep.continuousPosition`` attribute, i.e. tap step voltage increment, per step position. (integer)"""

        self.hasRegulatingControl = hasRegulatingControl
        """The *RatioTapChanger*'s ``RegulatingControl`` reference, false if unavailable (boolean)"""

        # RC stands for RegulatingControl
        self.RC_discrete = RC_discrete
        """The ``RegulatingControl``'s ``discrete`` attribute (integer)"""

        self.RC_mode = RC_mode
        """The ``RegulatingControl``'s ``mode`` attribute (string)"""

        self.RC_targetRange_kV = RC_targetRange_kV
        """The ``RegulatingControl``'s ``targetRange`` attribute (kV)"""

        self.RC_targetValue_kV = RC_targetValue_kV
        """The ``RegulatingControl``'s ``targetValue`` attribute (kV)"""


class Topology_NodeBreaker(object):
    """Additional data from the CIM files about the NodeBreaker topology of the system"""

    def __init__(self, CIM_filenames):
        self.CIM_filenames = CIM_filenames
        """List of the CIM files which were passed to :py:meth:`CIM2Matpower.import` (list if string)"""

        self.nodes = []
        """List of :py:class:`~Node`-s within the biggest ``TopologicalIsland`` of the imported power system."""

        self.substations = []
        """List of :py:class:`Topology_NodeBreaker.Substation`-s within the biggest ``TopologicalIsland`` of the imported power system."""

        self.areas = []
        """List of all :py:class:`~Area`-s within the imported CIM files."""

        self.zones = []
        """List of all :py:class:`Topology_NodeBreaker.Zone`-s within the imported CIM files."""

        self.switches = []
        """List of all :py:class:`Topology_NodeBreaker.Switch`-es within the biggest ``TopologicalIsland`` of the imported power system."""

        self.branches = []
        """List of all :py:class:`Topology_NodeBreaker.Branch`-es within the biggest ``TopologicalIsland`` of the imported power system."""

        self.generators = []
        """List of all :py:class:`Topology_NodeBreaker.Generator`-es within the biggest ``TopologicalIsland`` of the imported power system."""

        self.loads = []
        """List of all :py:class:`Topology_NodeBreaker.Load`-es within the biggest ``TopologicalIsland`` of the imported power system."""

        self.shunts = []
        """List of all :py:class:`Topology_NodeBreaker.Shunt`-es within the biggest ``TopologicalIsland`` of the imported power system."""

        self.phasetapchangers = []
        """List of all :py:class:`Topology_NodeBreaker.PhaseTapChanger`-s within the biggest ``TopologicalIsland`` of the imported power system."""

        self.ratiotapchangers = []
        """List of all :py:class:`Topology_NodeBreaker.RatioTapChanger`-s within the biggest ``TopologicalIsland`` of the imported power system."""