"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.writeSorted = exports.applyAndWrite = exports.applyPatchSet = exports.evaluatePatchSet = exports.loadPatchSet = void 0;
/**
 * Apply a JSON patch set into the given target file
 *
 * The sources can be taken from one or more directories.
 */
const path = require("path");
const fastJsonPatch = require("fast-json-patch");
const fs = require("fs-extra");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const sortJson = require('sort-json');
async function loadPatchSet(sourceDirectory, relativeTo = process.cwd()) {
    const ret = {};
    const files = await fs.readdir(sourceDirectory);
    for (const file of files) {
        const fullFile = path.join(sourceDirectory, file);
        const relName = path.relative(relativeTo, fullFile);
        if (file.startsWith('.')) {
            // Nothing, ignore
        }
        else if ((await fs.stat(fullFile)).isDirectory()) {
            ret[relName] = {
                type: 'set',
                sources: await loadPatchSet(fullFile, sourceDirectory),
            };
        }
        else if (file.endsWith('.json')) {
            ret[relName] = {
                type: file.indexOf('patch') === -1 ? 'fragment' : 'patch',
                data: await fs.readJson(fullFile),
            };
        }
    }
    return ret;
}
exports.loadPatchSet = loadPatchSet;
function evaluatePatchSet(sources, options = {}) {
    const targetObject = {};
    for (const key of Object.keys(sources).sort()) {
        const value = sources[key];
        switch (value.type) {
            case 'fragment':
                log(key);
                merge(targetObject, value.data, []);
                break;
            case 'patch':
                patch(targetObject, value.data, (m) => log(`${key}: ${m}`));
                break;
            case 'set':
                const evaluated = evaluatePatchSet(value.sources, options);
                log(key);
                merge(targetObject, evaluated, []);
                break;
        }
    }
    return targetObject;
    function log(x) {
        if (!options.quiet) {
            // eslint-disable-next-line no-console
            console.log(x);
        }
    }
}
exports.evaluatePatchSet = evaluatePatchSet;
/**
 * Load a patch set from a directory
 */
async function applyPatchSet(sourceDirectory, options = {}) {
    const patches = await loadPatchSet(sourceDirectory);
    return evaluatePatchSet(patches, options);
}
exports.applyPatchSet = applyPatchSet;
/**
 * Load a patch set and write it out to a file
 */
async function applyAndWrite(targetFile, sourceDirectory, options = {}) {
    const model = await applyPatchSet(sourceDirectory, options);
    await writeSorted(targetFile, model);
}
exports.applyAndWrite = applyAndWrite;
async function writeSorted(targetFile, data) {
    await fs.mkdirp(path.dirname(targetFile));
    await fs.writeJson(targetFile, sortJson(data), { spaces: 2 });
}
exports.writeSorted = writeSorted;
function printSorted(data) {
    process.stdout.write(JSON.stringify(sortJson(data), undefined, 2));
}
function merge(target, fragment, jsonPath) {
    if (!fragment) {
        return;
    }
    if (!target || typeof target !== 'object' || Array.isArray(target)) {
        throw new Error(`Expected object, found: '${target}' at '$.${jsonPath.join('.')}'`);
    }
    for (const key of Object.keys(fragment)) {
        if (key.startsWith('$')) {
            continue;
        }
        if (key in target) {
            const specVal = target[key];
            const fragVal = fragment[key];
            if (typeof specVal !== typeof fragVal) {
                // eslint-disable-next-line max-len
                throw new Error(`Attempted to merge ${JSON.stringify(fragVal)} into incompatible ${JSON.stringify(specVal)} at path ${jsonPath.join('/')}/${key}`);
            }
            if (specVal == fragVal) {
                continue;
            }
            if (typeof specVal !== 'object') {
                // eslint-disable-next-line max-len
                throw new Error(`Conflict when attempting to merge ${JSON.stringify(fragVal)} into ${JSON.stringify(specVal)} at path ${jsonPath.join('/')}/${key}`);
            }
            merge(specVal, fragVal, [...jsonPath, key]);
        }
        else {
            target[key] = fragment[key];
        }
    }
}
function patch(target, fragment, log) {
    var _a;
    if (!fragment) {
        return;
    }
    const patches = findPatches(target, fragment);
    for (const p of patches) {
        log((_a = p.description) !== null && _a !== void 0 ? _a : '');
        try {
            fastJsonPatch.applyPatch(target, p.operations);
        }
        catch (e) {
            throw new Error(`error applying patch: ${JSON.stringify(p, undefined, 2)}: ${e.message}`);
        }
    }
}
/**
 * Find the sets of patches to apply in a document
 *
 * Adjusts paths to be root-relative, which makes it possible to have paths
 * point outside the patch scope.
 */
function findPatches(data, patchSource) {
    const ret = [];
    recurse(data, patchSource, []);
    return ret;
    function recurse(actualData, fragment, jsonPath) {
        if (!fragment) {
            return;
        }
        if ('patch' in fragment) {
            const p = fragment.patch;
            if (!p.operations) {
                throw new Error(`Patch needs 'operations' key, got: ${JSON.stringify(p)}`);
            }
            ret.push({
                description: p.description,
                operations: p.operations.map((op) => adjustPaths(op, jsonPath)),
            });
        }
        else if ('patch:each' in fragment) {
            const p = fragment['patch:each'];
            if (typeof actualData !== 'object') {
                throw new Error(`Patch ${p.description}: expecting object in data, found '${actualData}'`);
            }
            if (!p.operations) {
                throw new Error(`Patch needs 'operations' key, got: ${JSON.stringify(p)}`);
            }
            for (const key in actualData) {
                ret.push({
                    description: `${key}: ${p.description}`,
                    operations: p.operations.map((op) => adjustPaths(op, [...jsonPath, key])),
                });
            }
        }
        else {
            for (const key of Object.keys(fragment)) {
                if (!(key in actualData)) {
                    actualData[key] = {};
                }
                recurse(actualData[key], fragment[key], [...jsonPath, key]);
            }
        }
    }
    function adjustPaths(op, jsonPath) {
        return {
            ...op,
            ...op.path ? { path: adjustPath(op.path, jsonPath) } : undefined,
            ...op.from ? { from: adjustPath(op.from, jsonPath) } : undefined,
        };
    }
    /**
     * Adjust path
     *
     * '$/' means from the root, otherwise interpret as relative path.
     */
    function adjustPath(originalPath, jsonPath) {
        if (typeof originalPath !== 'string') {
            throw new Error(`adjustPath: expected string, got ${JSON.stringify(originalPath)}`);
        }
        if (originalPath.startsWith('$/')) {
            return originalPath.slice(1);
        }
        return jsonPath.map(p => `/${p}`).join('') + originalPath;
    }
}
/**
 * Run this file as a CLI tool, to apply a patch set from the command line
 */
async function main(args) {
    const quiet = eatArg('-q', args) || eatArg('--quiet', args);
    if (args.length < 1) {
        throw new Error('Usage: patch-set <DIR> [<FILE>]');
    }
    const [dir, targetFile] = args;
    const model = await applyPatchSet(dir, { quiet });
    if (targetFile) {
        await writeSorted(targetFile, model);
    }
    else {
        printSorted(model);
    }
}
function eatArg(arg, args) {
    for (let i = 0; i < args.length; i++) {
        if (args[i] === arg) {
            args.splice(i, 1);
            return true;
        }
    }
    return false;
}
if (require.main === module) {
    main(process.argv.slice(2)).catch(e => {
        process.exitCode = 1;
        // eslint-disable-next-line no-console
        console.error(e.message);
    });
}
//# sourceMappingURL=data:application/json;base64,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