# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['djangomockingbird']

package_data = \
{'': ['*']}

install_requires = \
['django>=2.2,<3.0']

setup_kwargs = {
    'name': 'djangomockingbird',
    'version': '1.0.5',
    'description': 'The fastest way to write the fastest Django unit tests',
    'long_description': "# Django Mockingbird: the fastest way to write the fastest Django unit tests\n\n\n![GitHub](https://img.shields.io/github/license/larsvonschaff/Django-mockingbird)\n![PyPI](https://img.shields.io/pypi/v/djangomockingbird)\n\n\n\n## 1. What is Django Mockingbird and why would I need it?\n\nUntil now, there were two options for writing tests for a Django application: either create objects in the database for every test or mock the database queries using [Unit test’s Mock](https://docs.python.org/3/library/unittest.mock.html). While the former is slow, the latter is complicated to write and read. Both add a lot of setup code to our tests. Django Mockingbird introduces a new way to write tests for Django, which is fast to run as well as simple to write.\n\n## 2. How does it work?\n\nIt works by creating a mock object which behaves exactly like the Django model, but does not execute any queries under the hood. It only takes one line of code to use it in your test. It is not meant to be used in place of frameworks like [Pytest](https://docs.pytest.org/en/6.2.x/), but to complement them.\n\n## 3. How do I use it?\n\n### Installation \n\n```python\npip install djangomockingbird\n```\n\n### Constructions the mocks\n\n```python\n\nfrom models import Model\nfrom djangomockingbird import make_mocks\n\nmy_mock = make_mocks(Model)\n\n```\nmy_mock is now an object that mimics Model's behaviour exactly.\n\n### Using the mocks \n\n\nThe most straightforward way is to monkeypatch the object directly using its absolute path. In this example function_to_test is a function that can containt any amount of queries involving Model. With Django Mockingbird only the code below is necessary for this kind of test to pass, no matter how complicated the queries.\n\n\n```python\nfrom djangomockingbird import make_mocks\nimport myapp\nimport function_to_test\n\ndef test_my_test_case():\n\n    myapp.myfile.Model = make_mocks(Model)\n    result = function_to_test()\n    #assertions here\n\n```\nA good way to use Django mockingbird is also with [Pytest's](https://docs.pytest.org/en/stable/) monkeypatch.\n\n\n```python\nimport pytest\nfrom djangomockingbird import make_mocks\nimport myapp\nimport function_to_test\n\ndef test_my_test_case(monkeypatch):\n    mock_model = make_mocks(Model)\n    monkeypatch.setattr('myapp.myfile.Model', mock_model)\n    result = function_to_test()\n    #assertions here\n\n```\n\n### Specifiying mock return data\n\nYou can specify the values of specific fields of the model object you are mocking. If you don’t empty strings will be returned. Construct a dictionary with field names as keys and desired returs as values and pass it to the 'specs' argument of make_mocks. If you try to specify a nonexisant field as a key an error will be thrown, but you can specify any kind of value you want.\n\n```python\n\nmock = make_mocks(Model, specs={'model_field':'desired_value'})\n\n```\n\n\n### Attention! Cases where returns must be specified: Model methods\n\nIf your model has custom methods and they are used by the test, you must specify their names and return data to the mock, otherwise your tests won't pass. \n\n```python\n\n model_method_specs = {'to_dict': {'': ''}}\n mock = make_mocks(Model, model_method_specs=model_method_specs)\n \n ```\n\n\n## 4. Is it production ready? Can I help make it better? \n\nThis is still a very new project, but is quite stable for the general use case. However, there are advanced use cases that are not yet supported, most notably [custom model managers](https://docs.djangoproject.com/en/3.1/topics/db/managers/#custom-managers). For those test cases you can try supplementing Django Mockingbird with your own code or other libraries. Because this tool is really just one elaborate mock model it is very flexible and plays well with pretty much anything.\n\nWe would appreciate you opening issues to bring any defects or oversights to light. Contributions are also kindly accepted - see more on the code arhitecture principles below if you are interested. \n\n## 5. Where can I read more details on the architecture?\n\nRead about the how functional programming principles were used in the library [here](http://www.cmdctrlesc.xyz/post/6) and on metaprogramming features [here](http://www.cmdctrlesc.xyz/post/5).\n\n\n\n\n\n\n\n",
    'author': 'larsvonschaff',
    'author_email': 'larsvonschaff@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/larsvonschaff/Django-mockingbird',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
