"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkMultipleTargetGroupsEc2Service = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const network_multiple_target_groups_service_base_1 = require("../base/network-multiple-target-groups-service-base");
/**
 * An EC2 service running on an ECS cluster fronted by a network load balancer.
 */
class NetworkMultipleTargetGroupsEc2Service extends network_multiple_target_groups_service_base_1.NetworkMultipleTargetGroupsServiceBase {
    /**
     * Constructs a new instance of the NetworkMultipleTargetGroupsEc2Service class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        if (props.taskDefinition && props.taskImageOptions) {
            throw new Error('You must specify only one of TaskDefinition or TaskImageOptions.');
        }
        else if (props.taskDefinition) {
            this.taskDefinition = props.taskDefinition;
        }
        else if (props.taskImageOptions) {
            const taskImageOptions = props.taskImageOptions;
            this.taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'TaskDef', {
                executionRole: taskImageOptions.executionRole,
                taskRole: taskImageOptions.taskRole,
            });
            const containerName = taskImageOptions.containerName !== undefined ? taskImageOptions.containerName : 'web';
            const container = this.taskDefinition.addContainer(containerName, {
                image: taskImageOptions.image,
                cpu: props.cpu,
                memoryLimitMiB: props.memoryLimitMiB,
                memoryReservationMiB: props.memoryReservationMiB,
                environment: taskImageOptions.environment,
                secrets: taskImageOptions.secrets,
                logging: this.logDriver,
            });
            if (taskImageOptions.containerPorts) {
                for (const containerPort of taskImageOptions.containerPorts) {
                    container.addPortMappings({
                        containerPort,
                    });
                }
            }
        }
        else {
            throw new Error('You must specify one of: taskDefinition or image');
        }
        if (!this.taskDefinition.defaultContainer) {
            throw new Error('At least one essential container must be specified');
        }
        if (this.taskDefinition.defaultContainer.portMappings.length === 0) {
            this.taskDefinition.defaultContainer.addPortMappings({
                containerPort: 80,
            });
        }
        this.service = this.createEc2Service(props);
        if (props.targetGroups) {
            this.addPortMappingForTargets(this.taskDefinition.defaultContainer, props.targetGroups);
            this.targetGroup = this.registerECSTargets(this.service, this.taskDefinition.defaultContainer, props.targetGroups);
        }
        else {
            this.targetGroup = this.listener.addTargets('ECS', {
                targets: [this.service],
                port: this.taskDefinition.defaultContainer.portMappings[0].containerPort,
            });
        }
    }
    createEc2Service(props) {
        return new aws_ecs_1.Ec2Service(this, 'Service', {
            cluster: this.cluster,
            desiredCount: this.desiredCount,
            taskDefinition: this.taskDefinition,
            assignPublicIp: false,
            serviceName: props.serviceName,
            healthCheckGracePeriod: props.healthCheckGracePeriod,
            propagateTags: props.propagateTags,
            enableECSManagedTags: props.enableECSManagedTags,
            cloudMapOptions: props.cloudMapOptions,
        });
    }
}
exports.NetworkMultipleTargetGroupsEc2Service = NetworkMultipleTargetGroupsEc2Service;
//# sourceMappingURL=data:application/json;base64,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